poll.estimation <-
function (poll.information) 
{
# poll.estimation
#
# Supplementary materials for 
# Sturgis, P., Kuha, J., Baker, N., Callegaro, M., Fisher, S., Green, J., 
# Jennings, W., Lauderdale, B. E., and Smith, P. (2017)   
# `An assessment of the causes of the errors In the 2015 
# UK General Election opinion polls
#
# Function for calculating point estimates of cote shares
# Please see the article and an explanatory note in the supplementary materials for further information 
#
	require(survey)
#
	data <- poll.information$data
	weighting.margins <- poll.information$weighting.margins
	weighting.targets <- poll.information$weighting.targets
	LTV.variable <- poll.information$LTV.variable
	Vote.variable <- poll.information$Vote.variable	
	parties <- poll.information$parties
#
	svy.unweighted <- svydesign(ids=~1,data=data)
#
# Calibration (raking) weighting:
#
	svy.weighted <- rake(svy.unweighted, 
				sample.margins = weighting.margins, 
				population.margins = weighting.targets
			)

	svy.weighted$variables <- data.frame(svy.weighted$variables,Wraked=weights(svy.weighted))
	svy.weighted$variables$Wnew <- svy.weighted$variables$Wraked*svy.weighted$variables[,LTV.variable]
	data.tmp <- svy.weighted$variables
	data.tmp <- data.tmp[data.tmp[,Vote.variable] %in% parties,]
	svy.weighted <- svydesign(~1,data=data.tmp,weights=data.tmp$Wnew)
#
# Weighted point estimates 
#
	res.tmp <- prop.table(svytable(as.formula(paste("~",Vote.variable)),svy.weighted))
#
	res <- rep(NA,length(parties)+1)
	names(res) <- c("Con-Lab",parties)
	res[-1] <- res.tmp[names(res.tmp)%in% parties]
#
# Conservative-Labour difference
## This assumes that the labels for Conservatives and Labour in 
## Vote.variable include the strings "Con" and "Lab": 
#
	res[1] <- res[pmatch("Con",names(res))]-res[pmatch("Lab",names(res))] 
#
	return(100*res)
}
poll.bootstrap <-
function (poll.information, n.bootstrap=0, bootstrap.samples=NULL, alpha=0.95, bca=F) 
{
# poll.bootstrap
#
# Supplementary materials for 
# Sturgis, P., Kuha, J., Baker, N., Callegaro, M., Fisher, S., Green, J., 
# Jennings, W., Lauderdale, B. E., and Smith, P. (2017)   
# `An assessment of the causes of the errors In the 2015 
# UK General Election opinion polls
#
# Function for repeating estimation for resampled data from a poll
# and calculating boostrap standard errors and confidence intervals.
#
# Please see the article and an explanatory note in the supplementary materials for further information. 
#
#
# Function for calculating adjusted bootstrap percentile confidence intervals
# [adapted from the bcanon function in the bootstrap package (by Rob Tibshirani)]
	bootstrap.bca <- function(alpha){
		n <- nrow(poll.information$data)
		est.actual <- res.summary[,1]
		nboot <- nrow(res)	
		u <- matrix(0,n,length(est.actual))
		poll.information.tmp <- poll.information
		for (i in 1:n) {
			if(i%%200==0)cat("Bootstrap BCA jackknife (out of ",n, "):", i,date(),"\n")
			poll.information.tmp$data <- data[-i,]						
			u[i,] <- est.function(poll.information.tmp)
     		  }
#
		confpoints <- matrix(NA,ncol(u),2)
		dimnames(confpoints) <- list(names(est.actual), c("Lower","Upper"))
		for(party in names(est.actual)){
		      thetahat <- est.actual[party]
		      thetastar <- res[,party]
		      z0 <- qnorm(sum(thetastar < thetahat)/nboot)
			uu <- mean(u) - u
		      acc <- sum(uu * uu * uu)/(6 * (sum(uu * uu))^1.5)
	      	zalpha <- qnorm(c((1-alpha)/2,1-(1-alpha)/2))
		      tt <- pnorm(z0 + (z0 + zalpha)/(1 - acc * (z0 + zalpha)))
		      confpoints[party,] <- quantile(x = thetastar, probs = tt, type = 1)
		}
		return(confpoints)			
	}
#
#
	data <- poll.information$data
	quota.cols <- poll.information$quota.columns
	est.function <- poll.information$est.function
	res <- NULL
	ci <- NULL
#
# Estimates from the actual data
#
#
	if(!is.null(bootstrap.samples)){
		res <- bootstrap.samples$bootstrap.samples
		res.summary <- bootstrap.samples$summary
	}	
	else{
		est.actual <- est.function(poll.information)
		res.summary <- matrix(NA,length(est.actual),4)
		dimnames(res.summary) <- list(names(est.actual),c("Actual est.","BS Mean","BS Median","BS se")) 
		res.summary[,1] <- est.actual
	}
#
# Drawing bootstrap samples, if required
#	
	if(n.bootstrap>0 & is.null(bootstrap.samples)){
		res <- matrix(NA, n.bootstrap, length(est.actual)+1)
		colnames(res) <- c(names(est.actual),"N") 
		poll.information.tmp <- poll.information
		weighting.variables <- substring(as.character(poll.information$weighting.margins),2)
		for(j in seq(n.bootstrap)){
			if(j%%100==0)cat("Bootstrap iteration (out of", n.bootstrap, "):", j, date(),"\n")
			sample.again <- T
			while(sample.again){
				data.j <- quota.resample(data,quota.cols)
				check <- sapply(data.j[,weighting.variables],table)
				sample.again <- any(sapply(check,function(x){any(x==0)}))
			}
		poll.information.tmp$data <- data.j
		res[j,-ncol(res)] <- est.function(poll.information.tmp)
		res[j,ncol(res)] <- nrow(data.j)
		}	
		res.N <- res[,ncol(res)]
		res <- res[,-ncol(res)]
		res.summary[,2] <- colMeans(res)
		res[,which(apply(res,2,FUN=function(x){all(is.na(x))}))]<-0
		res.summary[,3] <- t(apply(res,2,quantile,probs=c(.5)))
		res.summary[,4] <- sqrt(apply(res,2,var))
		res <- cbind(res, N=res.N)
	}
#
# Bootstrap confidence intervals, if required
#
	if(!is.null(res)){
		n.ests <- nrow(res.summary)
		ci <- matrix(NA,n.ests,2*3)
		n.tmp <- paste(c("Normal.L","Normal.U","Percentile.L","Percentile.U","BCA.L","BCA.U"),100*alpha,sep="")
		dimnames(ci) <- list(rownames(res.summary),n.tmp)
		ci[,1] <- res.summary[,1]-qnorm(1-(1-alpha)/2)*sqrt(apply(res[,-ncol(res)],2,var))
		ci[,2] <- res.summary[,1]+qnorm(1-(1-alpha)/2)*sqrt(apply(res[,-ncol(res)],2,var))
		ci[,3:4] <- t(apply(res[,-ncol(res)],2,quantile,probs=c((1-alpha)/2,1-(1-alpha)/2)))
		if(bca)ci[,5:6] <- bootstrap.bca(alpha)
		ci <- round(ci,3)
	}
#
	list(summary=round(res.summary,3), conf.intervals=ci, bootstrap.samples=res)
}
quota.resample <-
function (data, quota.cols=NULL) 
{
# quota.resample
#
# Supplementary materials for 
# Sturgis, P., Kuha, J., Baker, N., Callegaro, M., Fisher, S., Green, J., 
# Jennings, W., Lauderdale, B. E., and Smith, P. (2017)   
# `An assessment of the causes of the errors In the 2015 
# UK General Election opinion polls
#
# Function for calculating point estimates of cote shares
# Please see the article and an explanatory note in the supplementary 
# materials for further information. 
#
# Function for resampling from a data frame to match totals of given variables
#
# Totals of quota variables
	quota.vars <- data[,quota.cols]
	quota.targets <- colSums(quota.vars)
#
	final.sample <- NULL
	continue <- T
	sample.from <- data
	n.sample <- nrow(data)
	quota.totals.sofar <- rep(0,length=ncol(data[,quota.cols]))
#
# Sampling:
#
	while(continue){
		new.sample <- sample.from[sample(seq(n.sample),n.sample,replace=T),,drop=F]
		quota.vars.sample <- new.sample[,quota.cols,drop=F]
		csums <- apply(quota.vars.sample,2,cumsum)
			if(nrow(quota.vars.sample)==1)csums <- matrix(csums,nrow=1)
		csums <- t(csums)+quota.totals.sofar
		not.over.quota <- apply(csums<=quota.targets,2,all)
		final.sample <- rbind(final.sample, new.sample[not.over.quota,,drop=F])
		quota.totals.sofar <- colSums(final.sample[,quota.cols])
		quota.targets.met <- quota.totals.sofar == quota.targets
		if(all(quota.targets.met))continue <- F # Stop if all quota targets met
		else{
			qvars.complete <- (sample.from[,quota.cols,drop=F])[,which(quota.targets.met),drop=F] 
			sample.from <- sample.from[rowSums(qvars.complete)==0,,drop=F]
			n.sample <- nrow(sample.from)
			if(n.sample==0)continue <- F # Stop is no more observations left to sample from
		}	
	}	
	final.sample
}
PollData <-
structure(list(ID = 1:2000, SexbyAge = structure(c(6L, 6L, 4L, 
6L, 3L, 6L, 4L, 3L, 7L, 2L, 5L, 3L, 2L, 8L, 8L, 8L, 8L, 3L, 6L, 
3L, 4L, 3L, 8L, 7L, 2L, 7L, 6L, 5L, 7L, 4L, 8L, 7L, 1L, 4L, 4L, 
5L, 3L, 8L, 8L, 1L, 1L, 4L, 6L, 6L, 7L, 6L, 1L, 5L, 7L, 6L, 7L, 
5L, 5L, 2L, 8L, 4L, 8L, 4L, 3L, 3L, 7L, 3L, 8L, 6L, 4L, 7L, 6L, 
5L, 7L, 6L, 4L, 7L, 8L, 4L, 6L, 6L, 7L, 4L, 7L, 7L, 2L, 8L, 4L, 
4L, 7L, 8L, 3L, 3L, 8L, 4L, 3L, 5L, 3L, 5L, 3L, 4L, 4L, 8L, 3L, 
6L, 8L, 1L, 7L, 6L, 6L, 4L, 7L, 3L, 4L, 3L, 8L, 4L, 6L, 8L, 2L, 
7L, 3L, 3L, 2L, 4L, 7L, 6L, 6L, 8L, 3L, 7L, 6L, 8L, 3L, 6L, 3L, 
6L, 4L, 2L, 6L, 7L, 2L, 3L, 5L, 7L, 4L, 2L, 4L, 3L, 7L, 7L, 1L, 
7L, 7L, 5L, 3L, 3L, 2L, 2L, 7L, 4L, 8L, 8L, 2L, 3L, 4L, 7L, 3L, 
7L, 4L, 6L, 8L, 7L, 8L, 7L, 3L, 7L, 7L, 6L, 1L, 3L, 3L, 7L, 4L, 
3L, 5L, 4L, 8L, 4L, 7L, 2L, 7L, 5L, 8L, 7L, 4L, 5L, 4L, 7L, 4L, 
6L, 3L, 2L, 4L, 7L, 7L, 3L, 8L, 1L, 3L, 6L, 8L, 7L, 5L, 2L, 7L, 
7L, 6L, 5L, 7L, 5L, 7L, 1L, 3L, 3L, 7L, 4L, 7L, 4L, 4L, 8L, 3L, 
4L, 1L, 3L, 6L, 8L, 6L, 7L, 6L, 7L, 6L, 4L, 3L, 6L, 3L, 8L, 3L, 
2L, 8L, 3L, 3L, 2L, 4L, 2L, 4L, 3L, 4L, 3L, 8L, 7L, 3L, 4L, 5L, 
7L, 8L, 8L, 7L, 8L, 2L, 4L, 4L, 6L, 8L, 4L, 6L, 3L, 7L, 4L, 8L, 
8L, 8L, 4L, 8L, 6L, 2L, 8L, 3L, 4L, 3L, 7L, 6L, 3L, 4L, 7L, 3L, 
3L, 7L, 4L, 4L, 4L, 6L, 7L, 1L, 5L, 2L, 3L, 7L, 3L, 8L, 7L, 1L, 
4L, 7L, 5L, 3L, 8L, 8L, 2L, 2L, 2L, 8L, 6L, 2L, 3L, 3L, 4L, 8L, 
8L, 4L, 2L, 4L, 8L, 1L, 8L, 3L, 2L, 1L, 1L, 7L, 7L, 8L, 3L, 8L, 
6L, 4L, 2L, 3L, 8L, 7L, 7L, 2L, 8L, 8L, 4L, 1L, 1L, 7L, 2L, 8L, 
3L, 3L, 7L, 3L, 8L, 8L, 3L, 8L, 8L, 3L, 4L, 8L, 7L, 7L, 3L, 6L, 
6L, 8L, 3L, 3L, 7L, 2L, 8L, 2L, 4L, 3L, 4L, 2L, 6L, 7L, 8L, 8L, 
6L, 4L, 7L, 4L, 1L, 6L, 1L, 8L, 7L, 6L, 1L, 3L, 8L, 8L, 2L, 3L, 
4L, 7L, 5L, 3L, 6L, 7L, 7L, 6L, 2L, 4L, 4L, 6L, 7L, 8L, 4L, 6L, 
7L, 8L, 8L, 1L, 8L, 6L, 7L, 8L, 8L, 2L, 8L, 2L, 7L, 6L, 5L, 7L, 
1L, 6L, 3L, 6L, 7L, 7L, 7L, 6L, 4L, 4L, 7L, 8L, 3L, 3L, 3L, 3L, 
7L, 8L, 3L, 2L, 8L, 3L, 7L, 8L, 2L, 4L, 3L, 3L, 3L, 1L, 3L, 3L, 
4L, 7L, 3L, 5L, 2L, 3L, 6L, 3L, 5L, 3L, 7L, 3L, 5L, 2L, 5L, 8L, 
7L, 8L, 3L, 6L, 8L, 2L, 3L, 4L, 3L, 6L, 7L, 5L, 3L, 4L, 7L, 7L, 
3L, 5L, 4L, 7L, 8L, 4L, 6L, 7L, 8L, 6L, 3L, 6L, 8L, 7L, 4L, 8L, 
2L, 8L, 5L, 3L, 3L, 2L, 4L, 2L, 6L, 4L, 5L, 8L, 8L, 8L, 7L, 4L, 
3L, 4L, 7L, 8L, 3L, 1L, 3L, 3L, 5L, 8L, 4L, 3L, 8L, 5L, 2L, 6L, 
8L, 8L, 2L, 3L, 8L, 6L, 5L, 3L, 7L, 2L, 5L, 4L, 1L, 8L, 8L, 3L, 
3L, 2L, 3L, 1L, 6L, 4L, 8L, 8L, 3L, 4L, 2L, 4L, 4L, 4L, 7L, 3L, 
8L, 8L, 3L, 7L, 5L, 4L, 8L, 6L, 7L, 8L, 7L, 8L, 3L, 3L, 7L, 4L, 
4L, 8L, 2L, 8L, 4L, 6L, 3L, 8L, 7L, 4L, 8L, 3L, 3L, 6L, 7L, 7L, 
8L, 8L, 3L, 3L, 7L, 7L, 4L, 7L, 8L, 6L, 8L, 3L, 2L, 2L, 4L, 4L, 
1L, 4L, 6L, 2L, 4L, 2L, 3L, 8L, 7L, 6L, 3L, 5L, 3L, 7L, 3L, 8L, 
8L, 3L, 6L, 4L, 8L, 6L, 8L, 7L, 3L, 7L, 8L, 3L, 5L, 8L, 4L, 8L, 
4L, 7L, 3L, 7L, 8L, 6L, 4L, 8L, 7L, 3L, 2L, 2L, 6L, 4L, 6L, 7L, 
8L, 4L, 3L, 8L, 2L, 6L, 4L, 5L, 5L, 6L, 2L, 6L, 4L, 2L, 4L, 3L, 
8L, 6L, 3L, 7L, 8L, 8L, 8L, 7L, 6L, 2L, 1L, 6L, 3L, 5L, 8L, 1L, 
5L, 5L, 2L, 2L, 7L, 4L, 8L, 6L, 8L, 7L, 3L, 7L, 4L, 3L, 2L, 3L, 
4L, 3L, 7L, 3L, 8L, 4L, 6L, 4L, 8L, 6L, 4L, 8L, 6L, 4L, 3L, 3L, 
5L, 4L, 8L, 4L, 7L, 8L, 6L, 2L, 5L, 3L, 4L, 3L, 7L, 5L, 1L, 3L, 
7L, 7L, 6L, 4L, 8L, 1L, 8L, 6L, 8L, 3L, 8L, 3L, 7L, 4L, 8L, 7L, 
3L, 8L, 4L, 6L, 7L, 6L, 7L, 7L, 4L, 8L, 4L, 7L, 6L, 4L, 4L, 8L, 
3L, 7L, 1L, 4L, 4L, 8L, 7L, 3L, 3L, 3L, 3L, 7L, 4L, 4L, 4L, 4L, 
4L, 3L, 6L, 5L, 3L, 7L, 2L, 7L, 7L, 4L, 3L, 3L, 7L, 1L, 7L, 7L, 
5L, 8L, 7L, 7L, 3L, 7L, 6L, 2L, 5L, 4L, 4L, 4L, 3L, 3L, 5L, 4L, 
3L, 4L, 7L, 4L, 8L, 3L, 7L, 2L, 7L, 3L, 4L, 4L, 3L, 3L, 2L, 8L, 
3L, 3L, 8L, 6L, 1L, 5L, 6L, 2L, 1L, 7L, 8L, 4L, 3L, 3L, 6L, 5L, 
4L, 2L, 6L, 7L, 1L, 4L, 6L, 7L, 8L, 6L, 2L, 3L, 4L, 6L, 2L, 5L, 
4L, 7L, 3L, 6L, 2L, 3L, 6L, 8L, 6L, 2L, 6L, 7L, 1L, 8L, 3L, 5L, 
3L, 3L, 3L, 3L, 2L, 4L, 8L, 7L, 2L, 7L, 7L, 5L, 4L, 8L, 4L, 6L, 
7L, 2L, 8L, 3L, 4L, 5L, 3L, 7L, 6L, 6L, 7L, 2L, 8L, 7L, 8L, 5L, 
4L, 6L, 6L, 3L, 8L, 6L, 3L, 3L, 8L, 2L, 2L, 4L, 8L, 3L, 4L, 7L, 
4L, 4L, 8L, 2L, 8L, 8L, 7L, 3L, 6L, 6L, 7L, 7L, 7L, 3L, 4L, 7L, 
7L, 7L, 4L, 8L, 2L, 2L, 5L, 7L, 3L, 3L, 3L, 8L, 8L, 4L, 4L, 3L, 
2L, 7L, 4L, 8L, 3L, 3L, 3L, 3L, 8L, 6L, 5L, 7L, 4L, 4L, 8L, 2L, 
3L, 4L, 8L, 8L, 4L, 3L, 7L, 4L, 2L, 8L, 2L, 1L, 8L, 7L, 5L, 2L, 
8L, 8L, 3L, 7L, 8L, 8L, 3L, 6L, 3L, 4L, 7L, 3L, 3L, 3L, 7L, 6L, 
4L, 4L, 4L, 3L, 7L, 6L, 6L, 2L, 7L, 6L, 8L, 3L, 8L, 8L, 4L, 8L, 
4L, 4L, 7L, 4L, 7L, 3L, 4L, 3L, 8L, 2L, 6L, 3L, 7L, 4L, 5L, 7L, 
8L, 7L, 8L, 3L, 8L, 8L, 5L, 4L, 5L, 6L, 5L, 3L, 4L, 3L, 8L, 3L, 
2L, 7L, 5L, 8L, 8L, 3L, 3L, 4L, 3L, 7L, 2L, 3L, 5L, 6L, 2L, 3L, 
2L, 3L, 7L, 8L, 8L, 7L, 7L, 8L, 4L, 6L, 4L, 7L, 4L, 8L, 4L, 5L, 
8L, 8L, 3L, 4L, 8L, 4L, 7L, 3L, 7L, 7L, 1L, 3L, 7L, 2L, 6L, 4L, 
4L, 5L, 4L, 3L, 2L, 7L, 4L, 4L, 3L, 6L, 6L, 6L, 1L, 5L, 6L, 6L, 
6L, 4L, 3L, 4L, 7L, 8L, 8L, 3L, 6L, 7L, 7L, 7L, 5L, 8L, 7L, 3L, 
3L, 6L, 1L, 6L, 8L, 7L, 7L, 3L, 1L, 3L, 2L, 3L, 7L, 3L, 7L, 1L, 
8L, 8L, 8L, 4L, 8L, 7L, 1L, 7L, 8L, 4L, 7L, 5L, 6L, 8L, 8L, 8L, 
4L, 4L, 3L, 2L, 7L, 1L, 7L, 6L, 7L, 2L, 3L, 3L, 8L, 8L, 2L, 5L, 
8L, 5L, 7L, 6L, 3L, 5L, 6L, 2L, 2L, 7L, 2L, 8L, 3L, 8L, 1L, 8L, 
6L, 8L, 7L, 3L, 3L, 3L, 7L, 2L, 4L, 3L, 8L, 8L, 6L, 3L, 7L, 4L, 
6L, 6L, 7L, 4L, 8L, 7L, 4L, 6L, 2L, 1L, 4L, 7L, 4L, 3L, 7L, 7L, 
7L, 2L, 4L, 1L, 5L, 7L, 3L, 3L, 8L, 2L, 1L, 7L, 4L, 3L, 5L, 8L, 
3L, 3L, 8L, 5L, 3L, 7L, 7L, 7L, 4L, 4L, 3L, 8L, 4L, 3L, 2L, 6L, 
4L, 7L, 3L, 8L, 3L, 5L, 8L, 4L, 5L, 3L, 3L, 7L, 5L, 7L, 1L, 8L, 
7L, 7L, 4L, 4L, 4L, 7L, 8L, 8L, 8L, 7L, 7L, 7L, 1L, 8L, 3L, 8L, 
1L, 8L, 1L, 8L, 6L, 2L, 7L, 8L, 3L, 1L, 8L, 4L, 6L, 2L, 8L, 4L, 
2L, 4L, 1L, 8L, 7L, 8L, 7L, 7L, 7L, 8L, 4L, 6L, 3L, 4L, 8L, 8L, 
6L, 4L, 3L, 3L, 4L, 2L, 1L, 8L, 6L, 4L, 8L, 6L, 7L, 2L, 6L, 3L, 
3L, 6L, 2L, 7L, 7L, 6L, 8L, 4L, 7L, 7L, 6L, 7L, 5L, 2L, 4L, 4L, 
8L, 5L, 8L, 3L, 3L, 7L, 2L, 1L, 2L, 8L, 1L, 8L, 7L, 3L, 7L, 7L, 
4L, 6L, 4L, 3L, 8L, 5L, 4L, 3L, 7L, 6L, 3L, 6L, 2L, 3L, 3L, 8L, 
3L, 8L, 8L, 4L, 6L, 7L, 1L, 7L, 2L, 1L, 7L, 2L, 4L, 5L, 7L, 7L, 
2L, 5L, 5L, 3L, 8L, 6L, 3L, 8L, 8L, 4L, 3L, 4L, 6L, 4L, 7L, 3L, 
2L, 3L, 7L, 4L, 2L, 3L, 6L, 4L, 8L, 6L, 6L, 7L, 4L, 7L, 7L, 2L, 
6L, 8L, 7L, 8L, 7L, 6L, 7L, 8L, 2L, 7L, 4L, 3L, 2L, 3L, 4L, 7L, 
7L, 8L, 4L, 7L, 8L, 7L, 3L, 6L, 5L, 8L, 2L, 2L, 6L, 7L, 7L, 5L, 
7L, 8L, 8L, 3L, 7L, 8L, 3L, 3L, 8L, 5L, 6L, 4L, 3L, 7L, 4L, 3L, 
7L, 3L, 6L, 3L, 4L, 8L, 3L, 1L, 2L, 3L, 3L, 6L, 7L, 7L, 7L, 5L, 
4L, 7L, 4L, 6L, 7L, 3L, 1L, 6L, 3L, 3L, 6L, 8L, 4L, 3L, 4L, 4L, 
5L, 3L, 6L, 8L, 3L, 5L, 7L, 8L, 8L, 3L, 7L, 6L, 4L, 8L, 4L, 3L, 
1L, 7L, 2L, 6L, 7L, 4L, 6L, 3L, 4L, 3L, 4L, 4L, 4L, 3L, 7L, 4L, 
7L, 7L, 7L, 6L, 3L, 7L, 2L, 8L, 8L, 3L, 2L, 8L, 3L, 4L, 3L, 4L, 
4L, 7L, 4L, 3L, 2L, 5L, 4L, 8L, 8L, 8L, 5L, 7L, 8L, 3L, 3L, 6L, 
3L, 5L, 4L, 4L, 1L, 3L, 7L, 3L, 1L, 3L, 3L, 4L, 5L, 8L, 3L, 7L, 
1L, 2L, 4L, 2L, 4L, 7L, 7L, 7L, 2L, 7L, 3L, 7L, 6L, 5L, 3L, 3L, 
3L, 2L, 8L, 7L, 7L, 7L, 7L, 3L, 6L, 4L, 4L, 5L, 3L, 7L, 2L, 7L, 
3L, 2L, 6L, 7L, 8L, 7L, 7L, 6L, 2L, 7L, 4L, 8L, 6L, 4L, 3L, 4L, 
8L, 6L, 2L, 6L, 2L, 3L, 4L, 1L, 2L, 5L, 3L, 7L, 4L, 3L, 7L, 3L, 
7L, 2L, 7L, 7L, 3L, 7L, 3L, 6L, 7L, 7L, 5L, 2L, 7L, 7L, 6L, 4L, 
5L, 4L, 3L, 4L, 2L, 8L, 3L, 7L, 1L, 6L, 7L, 3L, 3L, 8L, 8L, 4L, 
2L, 3L, 6L, 5L, 2L, 2L, 6L, 6L, 3L, 8L, 4L, 8L, 8L, 8L, 3L, 2L, 
8L, 8L, 8L, 4L, 8L, 4L, 3L, 4L, 4L, 4L, 3L, 8L, 6L, 1L, 3L, 4L, 
5L, 7L, 7L, 6L, 6L, 8L, 7L, 6L, 8L, 4L, 3L, 7L, 7L, 1L, 4L, 4L, 
4L, 8L, 3L, 7L, 3L, 2L, 7L, 7L, 4L, 4L, 2L, 3L, 3L, 1L, 7L, 5L, 
3L, 5L, 7L, 7L, 7L, 8L, 4L, 8L, 7L, 5L, 7L, 8L, 3L, 1L, 7L, 8L, 
7L, 6L, 3L, 8L, 7L, 4L, 3L, 8L, 6L, 8L, 3L, 7L, 6L, 2L, 6L, 4L, 
8L, 5L, 7L, 3L, 8L, 3L, 4L, 8L, 5L, 6L, 3L, 4L, 7L, 3L, 1L, 8L, 
5L, 7L, 8L, 3L, 3L, 3L, 5L, 4L, 4L, 8L, 4L, 4L, 2L, 3L, 5L, 8L, 
6L, 3L, 7L, 8L, 1L, 5L, 3L, 2L, 1L, 8L, 6L, 7L, 2L, 7L, 8L, 6L, 
3L, 2L, 6L, 6L, 3L, 3L, 2L, 7L, 7L, 4L, 6L, 7L, 5L, 3L, 3L, 7L, 
8L, 1L, 2L, 5L, 8L, 7L, 7L, 8L, 3L, 8L, 5L, 5L, 2L, 1L, 2L, 3L, 
8L, 2L, 3L, 3L, 2L, 7L, 8L, 7L, 4L, 7L, 7L, 5L, 3L, 8L, 6L, 5L, 
3L, 3L, 3L, 5L, 2L, 6L, 2L, 3L, 8L, 8L, 4L, 7L, 7L, 4L, 7L, 3L, 
7L, 3L, 2L, 5L, 8L, 8L, 7L, 6L, 5L, 2L, 8L, 8L, 4L, 1L, 3L, 3L, 
4L, 6L, 4L, 3L, 7L, 3L, 7L, 3L, 3L, 3L, 6L, 8L, 3L, 7L, 8L, 7L, 
7L, 6L, 5L, 4L, 3L, 3L, 1L, 1L, 3L, 5L, 4L, 6L, 5L, 7L, 6L, 6L, 
8L, 2L, 8L, 4L, 7L, 4L, 6L, 4L, 2L, 6L, 6L, 3L, 3L, 4L, 4L, 7L, 
8L, 4L, 2L, 4L, 6L, 3L, 4L, 4L, 7L, 3L, 1L, 8L, 4L), .Label = c("Male 18-24", 
"Male 25-39", "Male 40-59", "Male 60+", "Female 18-24", "Female 25-39", 
"Female 40-59", "Female 60+"), class = "factor"), Region = structure(c(7L, 
1L, 1L, 3L, 6L, 4L, 2L, 1L, 5L, 1L, 2L, 5L, 6L, 5L, 5L, 5L, 6L, 
3L, 3L, 5L, 7L, 2L, 1L, 4L, 1L, 3L, 4L, 1L, 5L, 2L, 7L, 2L, 5L, 
1L, 5L, 5L, 4L, 3L, 7L, 4L, 2L, 5L, 5L, 5L, 5L, 5L, 7L, 2L, 7L, 
6L, 2L, 3L, 2L, 1L, 5L, 2L, 6L, 5L, 4L, 7L, 2L, 5L, 5L, 4L, 2L, 
7L, 7L, 5L, 3L, 7L, 7L, 7L, 1L, 3L, 6L, 2L, 1L, 6L, 7L, 7L, 6L, 
1L, 2L, 2L, 4L, 5L, 5L, 2L, 1L, 7L, 5L, 7L, 1L, 1L, 7L, 2L, 4L, 
2L, 5L, 6L, 1L, 7L, 3L, 4L, 2L, 6L, 4L, 2L, 6L, 2L, 5L, 1L, 3L, 
1L, 2L, 7L, 5L, 7L, 5L, 3L, 6L, 1L, 7L, 7L, 2L, 6L, 3L, 3L, 2L, 
2L, 4L, 7L, 1L, 1L, 5L, 1L, 4L, 3L, 3L, 4L, 1L, 1L, 6L, 5L, 1L, 
1L, 3L, 1L, 4L, 7L, 6L, 4L, 3L, 6L, 1L, 6L, 4L, 6L, 3L, 5L, 1L, 
5L, 5L, 5L, 1L, 3L, 5L, 7L, 5L, 2L, 2L, 6L, 2L, 2L, 3L, 1L, 6L, 
1L, 1L, 4L, 5L, 6L, 5L, 3L, 2L, 4L, 3L, 6L, 7L, 1L, 1L, 5L, 1L, 
3L, 6L, 4L, 1L, 4L, 3L, 5L, 2L, 6L, 3L, 6L, 6L, 6L, 6L, 5L, 1L, 
5L, 2L, 7L, 5L, 6L, 2L, 7L, 5L, 5L, 2L, 1L, 6L, 1L, 4L, 2L, 6L, 
5L, 7L, 1L, 5L, 1L, 3L, 2L, 7L, 5L, 5L, 2L, 2L, 6L, 4L, 4L, 1L, 
6L, 1L, 3L, 2L, 2L, 7L, 5L, 5L, 2L, 5L, 4L, 7L, 6L, 1L, 2L, 5L, 
5L, 3L, 3L, 5L, 1L, 5L, 7L, 4L, 1L, 6L, 4L, 5L, 1L, 3L, 2L, 1L, 
1L, 2L, 5L, 7L, 1L, 2L, 3L, 5L, 1L, 1L, 4L, 1L, 2L, 5L, 5L, 7L, 
5L, 5L, 4L, 1L, 6L, 4L, 2L, 6L, 5L, 7L, 7L, 4L, 1L, 4L, 2L, 1L, 
3L, 7L, 5L, 1L, 6L, 1L, 2L, 6L, 1L, 1L, 4L, 5L, 4L, 3L, 5L, 6L, 
5L, 2L, 7L, 1L, 5L, 2L, 5L, 5L, 2L, 4L, 2L, 6L, 5L, 4L, 6L, 1L, 
6L, 5L, 2L, 7L, 5L, 4L, 5L, 4L, 6L, 1L, 2L, 1L, 3L, 6L, 2L, 7L, 
2L, 1L, 1L, 7L, 4L, 7L, 3L, 2L, 4L, 5L, 1L, 6L, 3L, 7L, 7L, 1L, 
7L, 5L, 2L, 1L, 1L, 3L, 3L, 3L, 5L, 5L, 6L, 3L, 3L, 7L, 5L, 1L, 
4L, 6L, 5L, 1L, 3L, 5L, 2L, 5L, 1L, 5L, 6L, 1L, 3L, 7L, 1L, 6L, 
7L, 3L, 5L, 2L, 3L, 7L, 7L, 3L, 5L, 5L, 5L, 2L, 7L, 1L, 5L, 2L, 
7L, 4L, 2L, 7L, 1L, 4L, 1L, 1L, 3L, 1L, 3L, 7L, 4L, 6L, 1L, 2L, 
6L, 3L, 2L, 5L, 1L, 5L, 2L, 4L, 7L, 5L, 5L, 2L, 4L, 3L, 3L, 7L, 
3L, 7L, 5L, 1L, 4L, 4L, 1L, 1L, 1L, 2L, 5L, 6L, 1L, 1L, 7L, 5L, 
1L, 6L, 5L, 1L, 1L, 5L, 5L, 5L, 1L, 7L, 2L, 7L, 1L, 2L, 5L, 7L, 
6L, 5L, 3L, 5L, 7L, 5L, 2L, 4L, 7L, 5L, 1L, 6L, 7L, 2L, 1L, 2L, 
2L, 1L, 5L, 5L, 5L, 5L, 2L, 5L, 4L, 3L, 1L, 5L, 3L, 5L, 1L, 2L, 
7L, 6L, 2L, 5L, 4L, 5L, 5L, 4L, 4L, 4L, 2L, 5L, 7L, 5L, 1L, 1L, 
7L, 5L, 5L, 3L, 2L, 2L, 3L, 3L, 7L, 5L, 7L, 5L, 4L, 3L, 6L, 7L, 
4L, 4L, 5L, 1L, 4L, 5L, 6L, 5L, 6L, 2L, 7L, 1L, 6L, 6L, 1L, 5L, 
7L, 7L, 6L, 2L, 7L, 1L, 1L, 7L, 2L, 5L, 2L, 2L, 5L, 5L, 6L, 1L, 
7L, 2L, 5L, 1L, 6L, 1L, 4L, 5L, 1L, 5L, 1L, 3L, 4L, 6L, 4L, 7L, 
5L, 2L, 1L, 3L, 7L, 6L, 3L, 5L, 1L, 1L, 4L, 5L, 6L, 5L, 4L, 4L, 
3L, 5L, 7L, 1L, 6L, 4L, 5L, 5L, 5L, 1L, 5L, 5L, 5L, 2L, 1L, 2L, 
7L, 1L, 5L, 5L, 4L, 5L, 3L, 1L, 7L, 1L, 4L, 7L, 3L, 4L, 2L, 3L, 
7L, 1L, 6L, 5L, 6L, 1L, 2L, 2L, 7L, 3L, 1L, 3L, 6L, 6L, 4L, 1L, 
5L, 4L, 3L, 2L, 4L, 5L, 4L, 4L, 1L, 2L, 6L, 7L, 5L, 3L, 5L, 1L, 
2L, 5L, 1L, 2L, 5L, 2L, 6L, 5L, 2L, 1L, 2L, 1L, 3L, 7L, 6L, 3L, 
3L, 4L, 5L, 2L, 7L, 1L, 5L, 2L, 1L, 1L, 7L, 2L, 1L, 4L, 4L, 5L, 
2L, 4L, 3L, 1L, 2L, 4L, 1L, 5L, 3L, 1L, 7L, 7L, 4L, 5L, 5L, 4L, 
1L, 4L, 3L, 7L, 2L, 1L, 2L, 7L, 7L, 5L, 3L, 4L, 2L, 5L, 5L, 2L, 
3L, 4L, 7L, 1L, 5L, 2L, 2L, 5L, 2L, 4L, 4L, 5L, 1L, 5L, 1L, 3L, 
1L, 1L, 1L, 7L, 1L, 1L, 2L, 5L, 2L, 2L, 5L, 4L, 4L, 7L, 2L, 6L, 
5L, 4L, 1L, 4L, 1L, 2L, 2L, 7L, 4L, 2L, 6L, 1L, 1L, 7L, 4L, 4L, 
5L, 5L, 1L, 2L, 3L, 7L, 3L, 4L, 5L, 7L, 5L, 7L, 7L, 5L, 4L, 5L, 
6L, 5L, 4L, 4L, 1L, 5L, 1L, 2L, 2L, 1L, 2L, 6L, 3L, 5L, 6L, 7L, 
2L, 3L, 2L, 5L, 2L, 1L, 6L, 3L, 3L, 7L, 6L, 3L, 2L, 2L, 1L, 6L, 
5L, 1L, 1L, 1L, 5L, 4L, 2L, 1L, 3L, 4L, 2L, 1L, 1L, 2L, 5L, 7L, 
1L, 3L, 6L, 5L, 2L, 6L, 4L, 3L, 6L, 1L, 1L, 1L, 7L, 7L, 1L, 1L, 
5L, 7L, 2L, 1L, 1L, 5L, 5L, 1L, 2L, 5L, 3L, 7L, 6L, 7L, 5L, 7L, 
3L, 7L, 5L, 1L, 7L, 3L, 1L, 4L, 4L, 3L, 2L, 4L, 2L, 6L, 6L, 3L, 
7L, 7L, 6L, 2L, 1L, 7L, 1L, 1L, 1L, 2L, 5L, 7L, 7L, 5L, 6L, 2L, 
5L, 6L, 6L, 1L, 5L, 7L, 5L, 3L, 2L, 7L, 5L, 1L, 4L, 4L, 5L, 6L, 
5L, 7L, 1L, 2L, 3L, 3L, 1L, 3L, 6L, 5L, 4L, 2L, 3L, 3L, 5L, 1L, 
5L, 3L, 6L, 3L, 6L, 2L, 7L, 5L, 7L, 7L, 1L, 4L, 5L, 6L, 3L, 2L, 
5L, 4L, 1L, 1L, 2L, 2L, 6L, 1L, 6L, 1L, 2L, 2L, 2L, 2L, 4L, 5L, 
6L, 2L, 2L, 7L, 7L, 1L, 2L, 6L, 1L, 5L, 5L, 6L, 2L, 5L, 1L, 2L, 
2L, 4L, 5L, 6L, 3L, 1L, 7L, 5L, 7L, 1L, 6L, 4L, 6L, 3L, 7L, 4L, 
3L, 1L, 5L, 6L, 6L, 2L, 5L, 1L, 2L, 1L, 1L, 1L, 5L, 1L, 5L, 6L, 
7L, 1L, 5L, 1L, 5L, 3L, 4L, 5L, 3L, 3L, 7L, 2L, 6L, 4L, 5L, 1L, 
5L, 1L, 1L, 5L, 7L, 5L, 5L, 2L, 5L, 7L, 2L, 7L, 1L, 4L, 2L, 7L, 
4L, 1L, 2L, 1L, 5L, 7L, 1L, 4L, 3L, 2L, 2L, 3L, 6L, 6L, 5L, 1L, 
5L, 1L, 1L, 6L, 2L, 1L, 5L, 4L, 3L, 2L, 1L, 5L, 7L, 5L, 7L, 5L, 
5L, 7L, 1L, 1L, 5L, 6L, 1L, 1L, 2L, 6L, 6L, 1L, 1L, 2L, 5L, 2L, 
1L, 5L, 1L, 1L, 1L, 7L, 6L, 3L, 5L, 4L, 2L, 7L, 1L, 2L, 5L, 5L, 
1L, 5L, 1L, 7L, 6L, 2L, 5L, 3L, 6L, 5L, 1L, 2L, 5L, 1L, 5L, 7L, 
7L, 7L, 1L, 2L, 6L, 1L, 5L, 1L, 2L, 1L, 1L, 7L, 6L, 7L, 5L, 5L, 
1L, 2L, 6L, 7L, 5L, 3L, 2L, 2L, 7L, 5L, 2L, 5L, 4L, 4L, 1L, 7L, 
3L, 3L, 5L, 5L, 5L, 5L, 1L, 1L, 2L, 4L, 1L, 6L, 4L, 2L, 5L, 3L, 
5L, 6L, 4L, 7L, 1L, 4L, 4L, 2L, 5L, 5L, 2L, 6L, 4L, 7L, 1L, 5L, 
5L, 5L, 1L, 5L, 1L, 4L, 6L, 4L, 1L, 2L, 7L, 4L, 1L, 5L, 7L, 3L, 
5L, 6L, 7L, 6L, 2L, 7L, 7L, 2L, 3L, 1L, 3L, 3L, 1L, 2L, 2L, 5L, 
5L, 5L, 2L, 5L, 7L, 3L, 6L, 2L, 1L, 5L, 2L, 7L, 1L, 3L, 3L, 1L, 
1L, 5L, 3L, 7L, 6L, 2L, 1L, 7L, 5L, 6L, 3L, 4L, 1L, 2L, 7L, 6L, 
5L, 1L, 7L, 2L, 4L, 6L, 1L, 6L, 5L, 2L, 5L, 3L, 2L, 5L, 6L, 7L, 
7L, 2L, 6L, 6L, 6L, 5L, 2L, 3L, 5L, 4L, 6L, 1L, 7L, 7L, 5L, 5L, 
5L, 6L, 7L, 1L, 2L, 4L, 2L, 6L, 5L, 3L, 3L, 4L, 4L, 2L, 7L, 6L, 
7L, 5L, 7L, 4L, 5L, 1L, 5L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 5L, 5L, 
7L, 5L, 1L, 5L, 1L, 1L, 5L, 2L, 7L, 5L, 3L, 2L, 2L, 5L, 6L, 1L, 
1L, 4L, 7L, 2L, 5L, 4L, 1L, 4L, 1L, 5L, 2L, 1L, 2L, 7L, 3L, 1L, 
5L, 4L, 7L, 1L, 5L, 1L, 7L, 2L, 1L, 6L, 1L, 2L, 5L, 2L, 1L, 3L, 
1L, 1L, 1L, 2L, 1L, 7L, 2L, 6L, 1L, 2L, 5L, 5L, 1L, 5L, 5L, 3L, 
1L, 6L, 1L, 1L, 5L, 6L, 1L, 4L, 3L, 4L, 1L, 6L, 7L, 7L, 6L, 1L, 
5L, 5L, 5L, 5L, 1L, 4L, 2L, 2L, 6L, 5L, 5L, 7L, 1L, 5L, 5L, 7L, 
4L, 5L, 1L, 7L, 7L, 7L, 2L, 1L, 2L, 1L, 6L, 5L, 2L, 5L, 6L, 3L, 
6L, 7L, 1L, 6L, 7L, 6L, 3L, 7L, 2L, 4L, 4L, 1L, 4L, 5L, 1L, 2L, 
1L, 6L, 1L, 5L, 5L, 1L, 5L, 4L, 4L, 1L, 3L, 3L, 1L, 7L, 3L, 5L, 
7L, 2L, 5L, 6L, 6L, 6L, 1L, 1L, 7L, 6L, 7L, 3L, 2L, 5L, 5L, 4L, 
2L, 4L, 1L, 4L, 3L, 3L, 6L, 5L, 1L, 7L, 7L, 5L, 4L, 3L, 3L, 3L, 
5L, 5L, 3L, 5L, 1L, 7L, 6L, 5L, 1L, 7L, 7L, 1L, 4L, 5L, 5L, 5L, 
1L, 2L, 3L, 4L, 2L, 6L, 5L, 5L, 2L, 1L, 2L, 5L, 1L, 5L, 4L, 7L, 
5L, 5L, 1L, 5L, 1L, 2L, 2L, 7L, 2L, 2L, 4L, 7L, 5L, 3L, 2L, 1L, 
1L, 6L, 5L, 5L, 5L, 3L, 2L, 1L, 5L, 2L, 1L, 3L, 5L, 1L, 3L, 7L, 
3L, 7L, 1L, 5L, 1L, 5L, 2L, 5L, 3L, 4L, 1L, 6L, 4L, 3L, 5L, 1L, 
2L, 5L, 1L, 5L, 5L, 1L, 2L, 2L, 1L, 1L, 1L, 5L, 6L, 2L, 1L, 6L, 
3L, 3L, 3L, 2L, 7L, 1L, 1L, 6L, 1L, 6L, 7L, 1L, 7L, 5L, 7L, 4L, 
2L, 6L, 5L, 3L, 1L, 3L, 1L, 5L, 5L, 7L, 5L, 4L, 1L, 6L, 7L, 1L, 
5L, 2L, 5L, 1L, 2L, 7L, 4L, 3L, 2L, 2L, 5L, 5L, 4L, 2L, 1L, 1L, 
5L, 5L, 1L, 2L, 2L, 1L, 5L, 1L, 1L, 2L, 1L, 2L, 2L, 3L, 1L, 2L, 
1L, 7L, 2L, 7L, 5L, 7L, 2L, 3L, 3L, 4L, 5L, 2L, 1L, 1L, 1L, 4L, 
1L, 1L, 6L, 4L, 5L, 5L, 6L, 5L, 7L, 2L, 5L, 4L, 5L, 5L, 5L, 5L, 
5L, 5L, 1L, 1L, 6L, 1L, 6L, 5L, 1L, 5L, 3L, 1L, 7L, 1L, 6L, 4L, 
1L, 3L, 1L, 2L, 5L, 1L, 2L, 1L, 7L, 4L, 4L, 5L, 6L, 1L, 1L, 7L, 
1L, 4L, 3L, 2L, 3L, 6L, 6L, 4L, 4L, 7L, 1L, 2L, 7L, 6L, 5L, 2L, 
2L, 1L, 7L, 1L, 1L, 4L, 5L, 5L, 5L, 1L, 2L, 6L, 7L, 7L, 4L, 3L, 
4L, 7L, 6L, 4L, 5L, 2L, 7L, 6L, 1L, 6L, 6L, 7L, 1L, 1L, 7L, 7L, 
4L, 4L, 5L, 5L, 1L, 3L, 4L, 2L, 1L, 5L, 1L, 4L, 7L, 5L, 7L, 4L, 
7L, 1L, 5L, 1L, 4L, 1L, 7L, 5L, 2L, 5L, 2L, 6L, 4L, 7L, 1L, 1L, 
7L, 5L, 7L, 4L, 1L, 5L, 5L, 2L, 1L, 5L, 5L, 4L, 6L, 4L, 3L, 7L, 
1L, 2L, 1L, 5L, 1L, 1L, 6L, 2L, 6L, 1L, 7L, 2L, 4L, 7L, 7L, 5L, 
5L, 5L, 6L, 5L, 1L, 5L, 1L, 2L, 4L, 3L, 1L, 7L, 4L, 2L, 7L, 1L, 
7L, 5L, 2L, 5L, 1L, 1L, 5L, 5L, 4L, 7L, 1L, 5L, 6L, 6L, 2L, 1L, 
6L, 1L, 3L, 2L, 6L, 5L, 5L, 1L, 4L, 3L, 7L, 7L, 6L, 4L, 7L, 5L, 
5L, 3L, 6L, 1L, 6L, 1L, 7L, 4L, 2L, 4L, 2L, 3L, 1L, 5L, 1L, 5L, 
5L, 5L, 7L, 6L, 5L, 4L, 6L, 5L, 1L, 4L, 1L, 1L, 5L, 2L, 5L, 2L, 
2L, 3L, 1L, 7L, 4L, 5L, 3L, 5L, 3L, 1L, 6L, 5L, 1L, 1L, 5L, 1L, 
5L, 1L, 7L, 5L, 2L, 7L, 2L, 7L, 6L, 4L, 7L, 7L, 5L, 2L, 5L, 5L, 
1L, 3L, 1L, 7L, 4L, 2L, 2L, 6L, 5L, 6L, 7L, 4L, 2L, 1L, 7L, 5L, 
7L, 1L, 3L, 7L, 2L, 2L, 2L, 1L, 4L, 5L, 1L, 1L, 6L, 1L, 5L, 1L, 
1L, 5L, 5L, 1L, 1L, 2L, 6L, 4L, 4L, 5L, 1L, 1L, 7L, 5L, 5L), .Label = c("North", 
"Midlands", "East", "London", "South", "Wales", "Scotland"), class = "factor"), 
    PartyID = structure(c(6L, 5L, 1L, 1L, 1L, 2L, 2L, 1L, 1L, 
    3L, 1L, 2L, 2L, 1L, 6L, 3L, 3L, 6L, 2L, 2L, 1L, 1L, 1L, 2L, 
    6L, 1L, 1L, 3L, 2L, 1L, 4L, 6L, 5L, 1L, 2L, 6L, 6L, 3L, 1L, 
    6L, 3L, 2L, 3L, 6L, 2L, 1L, 4L, 6L, 6L, 1L, 1L, 6L, 6L, 1L, 
    2L, 2L, 6L, 2L, 1L, 4L, 2L, 1L, 5L, 6L, 3L, 2L, 1L, 1L, 4L, 
    6L, 4L, 6L, 3L, 2L, 4L, 2L, 6L, 2L, 4L, 4L, 6L, 1L, 2L, 2L, 
    3L, 1L, 6L, 5L, 1L, 4L, 3L, 1L, 2L, 6L, 4L, 2L, 5L, 1L, 2L, 
    6L, 1L, 4L, 6L, 1L, 1L, 5L, 1L, 2L, 1L, 1L, 6L, 1L, 2L, 1L, 
    6L, 2L, 1L, 4L, 2L, 2L, 1L, 6L, 6L, 1L, 1L, 1L, 6L, 1L, 1L, 
    2L, 2L, 1L, 1L, 1L, 2L, 1L, 1L, 2L, 6L, 6L, 2L, 1L, 1L, 2L, 
    6L, 6L, 2L, 1L, 1L, 6L, 2L, 2L, 6L, 6L, 1L, 4L, 5L, 2L, 1L, 
    6L, 2L, 3L, 6L, 2L, 1L, 6L, 2L, 1L, 2L, 2L, 1L, 2L, 1L, 6L, 
    3L, 1L, 3L, 2L, 6L, 2L, 1L, 3L, 2L, 1L, 6L, 2L, 2L, 6L, 2L, 
    1L, 1L, 6L, 2L, 1L, 2L, 1L, 1L, 3L, 5L, 3L, 1L, 1L, 6L, 1L, 
    6L, 1L, 3L, 1L, 1L, 3L, 1L, 4L, 5L, 1L, 1L, 6L, 6L, 5L, 1L, 
    1L, 1L, 3L, 1L, 1L, 1L, 3L, 1L, 2L, 5L, 3L, 6L, 2L, 1L, 2L, 
    6L, 2L, 6L, 3L, 6L, 1L, 2L, 2L, 2L, 5L, 3L, 1L, 5L, 2L, 2L, 
    1L, 1L, 2L, 2L, 2L, 6L, 6L, 1L, 1L, 3L, 1L, 2L, 1L, 1L, 4L, 
    2L, 5L, 1L, 6L, 5L, 6L, 1L, 1L, 3L, 2L, 2L, 2L, 6L, 3L, 6L, 
    5L, 1L, 1L, 1L, 2L, 2L, 1L, 6L, 2L, 2L, 3L, 2L, 6L, 6L, 4L, 
    1L, 2L, 6L, 6L, 3L, 5L, 1L, 1L, 1L, 6L, 6L, 3L, 3L, 2L, 1L, 
    1L, 1L, 1L, 2L, 1L, 1L, 6L, 1L, 6L, 6L, 6L, 1L, 2L, 2L, 6L, 
    1L, 6L, 2L, 1L, 5L, 5L, 3L, 6L, 4L, 2L, 6L, 6L, 2L, 1L, 3L, 
    2L, 2L, 2L, 1L, 6L, 6L, 6L, 5L, 1L, 6L, 3L, 1L, 5L, 2L, 1L, 
    2L, 6L, 6L, 1L, 4L, 2L, 1L, 6L, 2L, 5L, 1L, 6L, 1L, 1L, 1L, 
    2L, 2L, 2L, 2L, 2L, 6L, 6L, 2L, 1L, 3L, 2L, 2L, 2L, 4L, 2L, 
    1L, 3L, 1L, 5L, 1L, 6L, 5L, 1L, 6L, 1L, 2L, 6L, 1L, 5L, 6L, 
    5L, 1L, 6L, 1L, 3L, 2L, 1L, 6L, 6L, 1L, 6L, 5L, 2L, 2L, 1L, 
    6L, 6L, 1L, 2L, 6L, 2L, 6L, 1L, 1L, 2L, 2L, 1L, 2L, 2L, 4L, 
    3L, 6L, 1L, 2L, 6L, 6L, 5L, 3L, 2L, 1L, 1L, 1L, 4L, 6L, 1L, 
    5L, 1L, 1L, 1L, 5L, 3L, 2L, 3L, 6L, 2L, 6L, 1L, 1L, 6L, 6L, 
    2L, 1L, 3L, 2L, 6L, 6L, 2L, 1L, 3L, 1L, 2L, 2L, 6L, 1L, 3L, 
    6L, 5L, 6L, 1L, 1L, 6L, 4L, 2L, 3L, 2L, 3L, 1L, 6L, 6L, 6L, 
    5L, 6L, 2L, 6L, 2L, 6L, 1L, 6L, 1L, 2L, 2L, 6L, 2L, 3L, 1L, 
    1L, 6L, 6L, 1L, 2L, 2L, 3L, 1L, 6L, 4L, 2L, 6L, 2L, 6L, 2L, 
    2L, 6L, 5L, 6L, 2L, 2L, 3L, 1L, 2L, 2L, 4L, 3L, 3L, 1L, 1L, 
    1L, 2L, 5L, 4L, 5L, 6L, 1L, 1L, 1L, 5L, 1L, 5L, 6L, 2L, 1L, 
    6L, 2L, 6L, 6L, 1L, 1L, 3L, 1L, 1L, 6L, 2L, 2L, 1L, 4L, 6L, 
    2L, 1L, 5L, 3L, 6L, 2L, 2L, 2L, 6L, 6L, 1L, 1L, 2L, 1L, 2L, 
    2L, 1L, 1L, 6L, 6L, 6L, 2L, 3L, 1L, 1L, 1L, 3L, 6L, 6L, 1L, 
    2L, 1L, 2L, 3L, 1L, 1L, 3L, 2L, 1L, 2L, 6L, 4L, 6L, 1L, 1L, 
    2L, 6L, 1L, 1L, 6L, 3L, 1L, 6L, 1L, 6L, 2L, 3L, 5L, 3L, 3L, 
    2L, 1L, 1L, 1L, 1L, 3L, 2L, 2L, 3L, 6L, 1L, 1L, 5L, 6L, 1L, 
    2L, 1L, 3L, 1L, 6L, 3L, 6L, 1L, 3L, 6L, 1L, 5L, 1L, 1L, 1L, 
    2L, 1L, 2L, 1L, 3L, 2L, 1L, 4L, 3L, 1L, 2L, 1L, 1L, 6L, 1L, 
    2L, 3L, 1L, 1L, 3L, 5L, 6L, 2L, 2L, 1L, 1L, 3L, 3L, 6L, 2L, 
    1L, 2L, 4L, 2L, 2L, 2L, 1L, 6L, 3L, 1L, 1L, 6L, 1L, 2L, 3L, 
    1L, 1L, 6L, 6L, 1L, 6L, 1L, 1L, 6L, 1L, 1L, 6L, 2L, 2L, 2L, 
    1L, 1L, 5L, 3L, 2L, 2L, 2L, 2L, 2L, 5L, 6L, 6L, 1L, 6L, 1L, 
    4L, 2L, 2L, 2L, 1L, 2L, 6L, 2L, 1L, 2L, 2L, 2L, 2L, 5L, 6L, 
    3L, 2L, 2L, 2L, 2L, 1L, 6L, 6L, 2L, 2L, 6L, 1L, 4L, 1L, 1L, 
    2L, 6L, 6L, 6L, 5L, 2L, 6L, 6L, 1L, 6L, 2L, 1L, 6L, 6L, 1L, 
    2L, 6L, 6L, 1L, 3L, 5L, 1L, 6L, 6L, 2L, 1L, 1L, 3L, 3L, 1L, 
    5L, 1L, 2L, 1L, 3L, 2L, 1L, 1L, 6L, 2L, 1L, 6L, 2L, 3L, 2L, 
    3L, 6L, 6L, 2L, 1L, 5L, 6L, 6L, 2L, 1L, 5L, 6L, 4L, 2L, 6L, 
    6L, 2L, 1L, 6L, 1L, 6L, 6L, 6L, 4L, 2L, 1L, 1L, 6L, 2L, 2L, 
    2L, 6L, 2L, 5L, 6L, 2L, 1L, 6L, 1L, 2L, 1L, 6L, 2L, 3L, 1L, 
    1L, 5L, 4L, 5L, 2L, 1L, 6L, 1L, 6L, 1L, 6L, 3L, 1L, 4L, 6L, 
    1L, 1L, 1L, 6L, 1L, 3L, 2L, 6L, 1L, 6L, 3L, 1L, 6L, 2L, 4L, 
    1L, 6L, 1L, 1L, 1L, 2L, 1L, 6L, 1L, 3L, 1L, 2L, 2L, 1L, 2L, 
    6L, 6L, 2L, 2L, 4L, 6L, 1L, 2L, 4L, 1L, 1L, 1L, 1L, 6L, 1L, 
    1L, 6L, 4L, 6L, 2L, 4L, 1L, 3L, 1L, 6L, 1L, 3L, 2L, 6L, 1L, 
    1L, 1L, 3L, 2L, 2L, 2L, 4L, 1L, 6L, 2L, 2L, 6L, 3L, 2L, 1L, 
    2L, 2L, 1L, 2L, 6L, 1L, 2L, 2L, 1L, 2L, 1L, 1L, 1L, 3L, 2L, 
    2L, 2L, 1L, 3L, 1L, 3L, 1L, 1L, 2L, 1L, 6L, 2L, 1L, 2L, 2L, 
    1L, 3L, 5L, 2L, 1L, 1L, 2L, 2L, 2L, 2L, 1L, 4L, 2L, 1L, 5L, 
    1L, 2L, 1L, 1L, 1L, 6L, 2L, 1L, 3L, 1L, 6L, 1L, 1L, 6L, 1L, 
    1L, 6L, 4L, 6L, 6L, 6L, 1L, 1L, 2L, 1L, 5L, 6L, 1L, 3L, 3L, 
    1L, 6L, 6L, 2L, 1L, 1L, 2L, 2L, 2L, 3L, 1L, 3L, 6L, 2L, 2L, 
    2L, 5L, 2L, 6L, 5L, 5L, 6L, 6L, 1L, 2L, 2L, 3L, 1L, 1L, 1L, 
    6L, 4L, 1L, 2L, 1L, 5L, 6L, 6L, 3L, 1L, 1L, 2L, 2L, 1L, 6L, 
    2L, 6L, 2L, 4L, 2L, 6L, 6L, 1L, 1L, 6L, 4L, 3L, 1L, 1L, 3L, 
    6L, 2L, 6L, 6L, 5L, 1L, 2L, 6L, 2L, 1L, 1L, 6L, 6L, 6L, 2L, 
    3L, 1L, 6L, 1L, 2L, 1L, 6L, 1L, 6L, 6L, 4L, 6L, 1L, 2L, 2L, 
    1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 5L, 1L, 1L, 6L, 6L, 1L, 1L, 
    3L, 2L, 1L, 2L, 2L, 6L, 6L, 2L, 2L, 1L, 1L, 2L, 2L, 1L, 3L, 
    3L, 1L, 4L, 3L, 6L, 5L, 2L, 1L, 1L, 1L, 3L, 2L, 2L, 2L, 1L, 
    2L, 6L, 6L, 2L, 1L, 6L, 1L, 6L, 1L, 6L, 1L, 1L, 6L, 6L, 2L, 
    2L, 1L, 1L, 2L, 1L, 2L, 3L, 6L, 3L, 3L, 1L, 6L, 1L, 3L, 6L, 
    2L, 2L, 2L, 6L, 1L, 5L, 1L, 6L, 6L, 5L, 1L, 1L, 1L, 2L, 1L, 
    6L, 5L, 1L, 2L, 6L, 2L, 2L, 1L, 6L, 3L, 6L, 1L, 1L, 5L, 1L, 
    1L, 6L, 1L, 1L, 1L, 2L, 3L, 6L, 1L, 1L, 4L, 6L, 6L, 6L, 6L, 
    2L, 6L, 2L, 5L, 2L, 1L, 2L, 2L, 2L, 1L, 2L, 6L, 3L, 6L, 2L, 
    1L, 5L, 1L, 2L, 1L, 6L, 1L, 6L, 2L, 6L, 1L, 3L, 2L, 1L, 1L, 
    6L, 6L, 6L, 6L, 6L, 2L, 2L, 3L, 6L, 6L, 6L, 6L, 2L, 1L, 6L, 
    5L, 3L, 1L, 6L, 6L, 6L, 2L, 6L, 6L, 6L, 3L, 4L, 2L, 2L, 1L, 
    5L, 2L, 3L, 1L, 6L, 2L, 1L, 6L, 1L, 1L, 6L, 1L, 2L, 2L, 1L, 
    6L, 1L, 6L, 2L, 1L, 2L, 5L, 6L, 3L, 1L, 6L, 2L, 6L, 6L, 4L, 
    1L, 6L, 2L, 6L, 1L, 1L, 5L, 1L, 6L, 5L, 1L, 6L, 3L, 6L, 3L, 
    1L, 3L, 1L, 2L, 1L, 3L, 2L, 2L, 1L, 1L, 2L, 5L, 2L, 6L, 2L, 
    5L, 6L, 1L, 6L, 3L, 2L, 6L, 2L, 1L, 2L, 6L, 6L, 1L, 1L, 3L, 
    2L, 1L, 1L, 3L, 6L, 6L, 3L, 1L, 3L, 6L, 1L, 1L, 3L, 6L, 6L, 
    3L, 5L, 1L, 2L, 2L, 1L, 6L, 1L, 1L, 3L, 2L, 3L, 1L, 2L, 2L, 
    1L, 2L, 5L, 3L, 2L, 1L, 2L, 5L, 2L, 5L, 2L, 6L, 5L, 1L, 6L, 
    6L, 4L, 6L, 1L, 2L, 6L, 2L, 6L, 2L, 6L, 2L, 6L, 2L, 6L, 2L, 
    2L, 6L, 1L, 6L, 2L, 4L, 2L, 3L, 6L, 6L, 1L, 6L, 6L, 6L, 1L, 
    2L, 1L, 3L, 2L, 2L, 6L, 1L, 2L, 4L, 2L, 6L, 4L, 6L, 2L, 1L, 
    6L, 2L, 1L, 2L, 6L, 1L, 2L, 2L, 6L, 1L, 6L, 2L, 2L, 6L, 6L, 
    1L, 1L, 1L, 2L, 2L, 2L, 4L, 5L, 3L, 5L, 1L, 2L, 1L, 1L, 2L, 
    2L, 6L, 2L, 1L, 5L, 2L, 2L, 2L, 1L, 5L, 6L, 6L, 1L, 2L, 2L, 
    2L, 6L, 6L, 2L, 2L, 1L, 6L, 1L, 2L, 2L, 2L, 1L, 1L, 2L, 2L, 
    1L, 5L, 6L, 2L, 2L, 2L, 4L, 1L, 3L, 6L, 3L, 6L, 2L, 5L, 2L, 
    6L, 2L, 6L, 1L, 2L, 6L, 1L, 2L, 6L, 1L, 2L, 5L, 6L, 3L, 6L, 
    2L, 1L, 6L, 6L, 1L, 1L, 6L, 2L, 1L, 6L, 2L, 6L, 6L, 2L, 5L, 
    2L, 3L, 6L, 6L, 6L, 2L, 2L, 2L, 2L, 6L, 2L, 5L, 1L, 5L, 4L, 
    2L, 1L, 6L, 3L, 1L, 1L, 1L, 6L, 6L, 1L, 1L, 1L, 6L, 5L, 5L, 
    5L, 3L, 1L, 1L, 2L, 6L, 1L, 6L, 6L, 6L, 1L, 2L, 1L, 6L, 6L, 
    1L, 1L, 3L, 3L, 6L, 6L, 3L, 2L, 6L, 1L, 2L, 1L, 6L, 1L, 4L, 
    2L, 6L, 2L, 1L, 6L, 5L, 5L, 2L, 6L, 1L, 1L, 2L, 4L, 2L, 6L, 
    5L, 1L, 2L, 1L, 2L, 3L, 6L, 2L, 2L, 6L, 1L, 1L, 3L, 1L, 3L, 
    6L, 2L, 6L, 1L, 1L, 5L, 5L, 1L, 1L, 5L, 6L, 5L, 1L, 6L, 6L, 
    6L, 2L, 1L, 6L, 6L, 1L, 2L, 2L, 6L, 1L, 3L, 6L, 6L, 5L, 2L, 
    6L, 6L, 1L, 1L, 1L, 2L, 2L, 1L, 2L, 1L, 1L, 3L, 6L, 5L, 5L, 
    3L, 1L, 2L, 1L, 6L, 2L, 2L, 1L, 5L, 6L, 6L, 1L, 2L, 2L, 1L, 
    5L, 2L, 2L, 1L, 6L, 1L, 1L, 1L, 1L, 1L, 6L, 1L, 1L, 3L, 1L, 
    6L, 1L, 1L, 1L, 2L, 2L, 1L, 1L, 2L, 4L, 6L, 1L, 2L, 2L, 2L, 
    4L, 2L, 2L, 1L, 3L, 3L, 2L, 4L, 1L, 2L, 5L, 5L, 1L, 4L, 2L, 
    1L, 1L, 2L, 1L, 6L, 2L, 6L, 2L, 1L, 1L, 6L, 6L, 1L, 2L, 5L, 
    6L, 2L, 1L, 1L, 6L, 1L, 1L, 1L, 1L, 2L, 3L, 5L, 6L, 2L, 6L, 
    3L, 6L, 2L, 2L, 1L, 2L, 1L, 2L, 1L, 6L, 1L, 6L, 5L, 1L, 4L, 
    1L, 1L, 1L, 1L, 6L, 4L, 2L, 3L, 2L, 2L, 2L, 3L, 4L, 6L, 5L, 
    6L, 2L, 1L, 2L, 6L, 6L, 2L, 1L, 2L, 2L, 1L, 6L, 1L, 1L, 3L, 
    4L, 5L, 2L, 6L, 6L, 6L, 6L, 1L, 1L, 1L, 5L, 4L, 3L, 5L, 6L, 
    4L, 6L, 2L, 5L, 2L, 1L, 6L, 2L, 5L, 2L, 2L, 2L, 2L, 1L, 1L, 
    1L, 4L, 1L, 1L, 2L, 1L, 6L, 5L, 1L, 1L, 1L, 1L, 1L, 1L, 3L, 
    1L, 6L, 6L, 1L, 1L, 2L, 6L, 5L, 6L, 6L, 2L, 2L, 6L, 5L, 1L, 
    2L, 2L, 3L, 6L, 1L, 6L, 2L, 6L, 1L, 2L, 6L, 1L, 6L, 2L, 3L, 
    3L, 6L, 5L, 6L, 2L, 2L, 3L, 1L, 1L, 3L, 6L, 6L, 2L, 1L, 6L, 
    2L, 2L, 6L, 6L, 2L, 6L, 6L, 6L, 1L, 1L, 1L, 1L, 6L, 3L, 2L, 
    1L, 6L, 5L, 6L, 2L, 1L, 1L, 5L, 6L, 5L, 1L, 6L, 6L, 1L, 6L, 
    6L, 1L, 6L, 2L, 1L, 3L, 5L, 6L, 2L, 1L, 2L, 1L, 1L, 6L, 1L, 
    6L, 4L, 5L, 2L, 4L, 6L, 1L, 1L, 5L, 2L, 1L, 2L, 1L, 2L, 1L, 
    2L, 1L, 1L, 6L, 6L, 5L, 6L, 1L, 2L, 1L, 1L, 1L, 6L, 2L, 5L, 
    1L, 2L, 2L, 4L, 3L, 6L, 1L, 2L, 5L, 1L, 2L), .Label = c("Lab", 
    "Con", "LibDem", "SNP or PC", "Other", "None or DK"), class = "factor"), 
    LTV = c(1, 1, 1, 1, 1, 1, 1, 0.7, 1, 1, 0.7, 0.9, 1, 0.9, 
    0, 0.8, 1, 0.9, 1, 0.9, 1, 0.9, 1, 1, 0.7, 1, 1, 1, 1, 0.9, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 
    0.4, 1, 1, 0.4, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 
    1, 1, 0.7, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
    1, 0.2, 0, 1, 1, 1, 1, 0.5, 0.6, 1, 1, 1, 1, 1, 0, 1, 0.8, 
    1, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 
    1, 0.9, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 
    1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0.8, 1, 1, 0, 1, 
    1, 1, 1, 1, 1, 0.9, 1, 0.9, 0.3, 1, 1, 0.8, 1, 0.9, 1, 1, 
    1, 0.9, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 1, 
    0.1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 0.6, 
    0.7, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 0.8, 1, 1, 0.9, 1, 1, 
    1, 1, 1, 1, 1, 0.7, 1, 1, 1, 0.9, 1, 1, 1, 0, 0.9, 1, 1, 
    1, 0.6, 0.9, 1, 1, 1, 1, 1, 0.9, 1, 1, 0.9, 1, 1, 1, 1, 0.8, 
    0.1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.8, 0, 0.8, 
    1, 1, 1, 1, 1, 1, 0.6, 0.7, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 0.7, 1, 1, 1, 1, 0.9, 0.9, 1, 0.2, 1, 1, 
    1, 1, 1, 0.9, 1, 1, 0, 1, 0.3, 1, 1, 1, 1, 1, 1, 1, 1, 0.9, 
    1, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 1, 0.5, 0.1, 
    1, 1, 0.8, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 1, 0.7, 1, 1, 
    1, 1, 1, 1, 1, 1, 0.8, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 0.8, 0, 0.8, 1, 1, 0.9, 
    1, 1, 0, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 0.9, 1, 0, 1, 1, 1, 
    0.9, 1, 1, 1, 1, 1, 0.8, 1, 1, 1, 1, 1, 0.5, 0.5, 1, 0.5, 
    1, 1, 0.5, 0.9, 1, 0.9, 1, 1, 0.9, 0.9, 1, 1, 0, 1, 1, 1, 
    0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9, 0.7, 1, 0.1, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 
    1, 0, 1, 1, 0.7, 0.9, 1, 0, 1, 0.7, 0.9, 1, 1, 1, 1, 1, 1, 
    1, 0, 1, 0.6, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 0.9, 1, 1, 0.9, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0.9, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 1, 0, 1, 1, 1, 1, 1, 1, 0, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.8, 1, 1, 1, 0.9, 1, 1, 0.9, 
    0.8, 1, 0, 1, 0, 1, 0.9, 1, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.7, 1, 1, 0.9, 
    1, 1, 1, 0.8, 1, 0.3, 1, 0.7, 1, 0.8, 1, 1, 1, 1, 0.9, 1, 
    1, 1, 1, 1, 0, 1, 1, 1, 1, 0.8, 1, 1, 1, 1, 0.9, 1, 0, 1, 
    1, 1, 0.1, 1, 1, 1, 1, 0.9, 0.9, 1, 1, 1, 1, 1, 0.8, 1, 1, 
    1, 1, 1, 1, 1, 0.5, 1, 0.9, 1, 1, 1, 1, 0.2, 1, 1, 0.8, 1, 
    1, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 0.5, 1, 1, 0.8, 
    1, 1, 1, 1, 0.6, 1, 0, 1, 1, 0, 0.9, 1, 1, 1, 1, 1, 0.9, 
    1, 1, 1, 1, 1, 0.8, 0.9, 1, 1, 0.8, 0.8, 1, 1, 1, 1, 1, 0.9, 
    0.9, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0.6, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 1, 0.6, 1, 1, 0.9, 1, 1, 1, 0.8, 1, 1, 
    1, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 0.5, 1, 0.9, 1, 0.8, 1, 1, 
    1, 1, 1, 1, 1, 0, 0.9, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 
    1, 1, 1, 0, 1, 1, 1, 0.6, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0.1, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0.8, 1, 1, 1, 
    1, 1, 0, 1, 0.9, 1, 1, 1, 0, 1, 0.2, 1, 1, 0.8, 1, 1, 1, 
    0.8, 1, 1, 1, 0.4, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.7, 1, 
    1, 0.9, 1, 1, 0.9, 1, 1, 1, 1, 0.7, 1, 1, 1, 0.2, 0.9, 1, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.7, 1, 1, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 0.9, 0, 1, 0.9, 1, 1, 1, 0, 1, 1, 1, 0.8, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.7, 1, 1, 1, 0.2, 0.9, 
    1, 0, 1, 1, 1, 1, 1, 1, 1, 0.8, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 0.8, 1, 1, 1, 1, 1, 0.9, 0.8, 1, 1, 0.1, 
    1, 1, 0.5, 1, 0.7, 1, 1, 1, 1, 1, 0.2, 1, 0.8, 1, 1, 1, 0, 
    1, 1, 1, 1, 1, 0.5, 1, 1, 1, 1, 1, 0.8, 1, 1, 1, 1, 1, 1, 
    0.6, 1, 1, 1, 1, 0.9, 0.5, 1, 1, 0.8, 1, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 0.9, 1, 1, 0.7, 1, 1, 1, 1, 1, 1, 0.9, 1, 
    0.8, 1, 1, 1, 1, 1, 1, 1, 1, 0.8, 1, 0.8, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 1, 1, 1, 0.3, 1, 1, 0.7, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.8, 1, 
    1, 1, 1, 1, 0.8, 0, 1, 0.8, 0.9, 1, 1, 0.1, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 1, 0.8, 1, 1, 0.8, 1, 1, 1, 1, 1, 1, 1, 
    0.7, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 0.5, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 0.8, 1, 1, 0.3, 1, 1, 1, 1, 0.3, 1, 1, 
    1, 0, 0.9, 0.8, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0.6, 
    1, 1, 1, 0.5, 0.2, 1, 1, 1, 1, 1, 1, 1, 1, 0.9, 0.9, 1, 1, 
    1, 1, 1, 0.9, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.3, 1, 0.9, 1, 
    1, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 0.7, 0.8, 0.8, 1, 1, 1, 1, 1, 1, 0.2, 1, 1, 1, 
    0, 1, 0.8, 1, 1, 0.7, 1, 1, 1, 1, 1, 0.2, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 0.9, 0.9, 1, 1, 0, 1, 1, 0.9, 1, 0, 1, 
    0, 1, 1, 1, 1, 1, 0.5, 1, 1, 1, 1, 0.2, 0.3, 1, 0, 1, 1, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 0.9, 1, 1, 1, 
    1, 1, 1, 1, 0, 0.8, 1, 1, 1, 1, 1, 1, 0.1, 1, 0, 1, 0.9, 
    1, 1, 0.9, 1, 1, 1, 1, 1, 1, 1, 0.7, 1, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
    0.6, 1, 1, 0, 1, 0.8, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 0.9, 1, 
    0.6, 0, 1, 1, 1, 1, 1, 1, 1, 0.9, 0.6, 1, 1, 0.1, 1, 0, 1, 
    1, 1, 1, 0.9, 1, 0.9, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 0, 1, 
    0.7, 1, 1, 1, 1, 0, 0.9, 1, 0, 0.9, 0.9, 1, 1, 1, 1, 1, 0.9, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.8, 1, 1, 1, 1, 1, 1, 1, 1, 
    0.9, 1, 0, 0.4, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.5, 1, 1, 1, 
    0.2, 1, 1, 1, 1, 0.5, 1, 1, 1, 1, 1, 1, 1, 1, 0.5, 1, 0, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.7, 1, 1, 0.9, 1, 1, 1, 0.9, 
    0.9, 0.9, 1, 0.9, 0.6, 1, 1, 1, 0.8, 1, 1, 1, 0, 1, 1, 1, 
    1, 1, 1, 0.7, 1, 1, 0.8, 1, 0.6, 1, 1, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 0.5, 0.9, 1, 1, 1, 1, 1, 1, 1, 0.6, 1, 1, 1, 
    1, 1, 1, 0, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 0.7, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 1, 1, 1, 0.7, 1, 
    1, 1, 0.9, 1, 1, 1, 0.2, 1, 1, 1, 1, 0.8, 1, 1, 1, 0.9, 1, 
    1, 1, 1, 1, 0, 1, 0.6, 1, 1, 0.8, 1, 1, 1, 1, 1, 1, 0.1, 
    0, 1, 1, 0.8, 0, 0.9, 0.7, 1, 1, 1, 1, 0.5, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 0.9, 1, 1, 1, 0.7, 1, 1, 1, 1, 1, 1, 0.8, 
    1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0.9, 1, 1, 1, 1, 0, 1, 1, 
    1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 0.9, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.5, 
    1, 1, 1, 1, 1, 1, 1, 0.9, 1, 1, 0.9, 0.9, 1, 0.9, 1, 1, 0.5, 
    1, 0.3, 0.5, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 0.7, 1, 0.9, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 0.5, 1, 1, 1, 1, 0.9, 1, 1, 0.9, 1, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.9, 1, 0.9, 1, 0.9, 
    1, 1, 1, 0, 1, 1, 1, 0.9, 1, 1, 1, 0.5, 1, 1, 1, 0.9, 0.9, 
    1, 1, 1, 1, 1, 0.9, 0, 1, 1, 1, 1, 1, 0.9, 0.8, 1, 1, 0.1, 
    0.9, 1, 1, 1, 1, 1, 1, 0.9, 1, 0.9, 0.7, 1, 1, 1, 1, 1, 1, 
    1, 0.3, 1, 1, 1, 1, 0, 0.9, 0.9, 1, 1, 1, 0.8, 1, 1, 1, 1, 
    1, 1, 1, 1, 0.9, 1, 1, 1, 0.4, 1, 1, 1, 0.5, 1, 1, 1, 0.7, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0.9, 1, 0, 1, 1, 0.9, 
    1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 0.9, 1, 0.6, 1, 1, 0.9, 
    1, 0.9, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
    0, 1, 1, 0.9, 1, 0.8, 1, 1, 1, 1, 1, 1, 1, 1, 0.8, 0.9, 0.9, 
    1, 1, 1, 1, 1, 1), VoteIntention = structure(c(6L, 8L, 2L, 
    4L, 2L, 1L, 1L, 2L, 2L, 3L, 2L, 1L, 2L, 2L, 9L, 3L, 3L, 9L, 
    1L, 1L, 6L, 2L, 3L, 1L, 5L, 2L, 2L, 5L, 1L, 1L, 6L, 9L, 1L, 
    4L, 1L, 2L, 5L, 1L, 2L, 9L, 3L, 4L, 1L, 9L, 1L, 2L, 6L, 9L, 
    6L, 1L, 2L, 2L, 9L, 2L, 1L, 1L, 4L, 1L, 3L, 6L, 1L, 2L, 4L, 
    9L, 4L, 1L, 6L, 2L, 9L, 8L, 6L, 6L, 3L, 1L, 7L, 1L, 3L, 4L, 
    9L, 6L, 3L, 9L, 1L, 1L, 3L, 2L, 9L, 9L, 9L, 4L, 3L, 2L, 9L, 
    9L, 6L, 4L, 4L, 2L, 1L, 9L, 4L, 6L, 3L, 5L, 1L, 4L, 2L, 1L, 
    2L, 2L, 3L, 2L, 1L, 2L, 3L, 9L, 2L, 6L, 1L, 1L, 2L, 2L, 2L, 
    2L, 2L, 3L, 2L, 2L, 2L, 1L, 1L, 2L, 2L, 2L, 1L, 2L, 2L, 1L, 
    2L, 9L, 1L, 1L, 1L, 1L, 2L, 9L, 1L, 2L, 9L, 1L, 1L, 1L, 2L, 
    1L, 2L, 9L, 1L, 1L, 2L, 3L, 1L, 3L, 9L, 3L, 1L, 9L, 3L, 2L, 
    1L, 1L, 3L, 3L, 2L, 1L, 3L, 2L, 3L, 1L, 9L, 1L, 5L, 3L, 1L, 
    1L, 9L, 1L, 1L, 2L, 1L, 5L, 2L, 9L, 1L, 2L, 1L, 2L, 2L, 1L, 
    4L, 3L, 2L, 2L, 1L, 2L, 2L, 2L, 1L, 2L, 9L, 9L, 9L, 6L, 4L, 
    3L, 8L, 6L, 8L, 3L, 2L, 2L, 2L, 3L, 5L, 2L, 2L, 5L, 6L, 1L, 
    2L, 2L, 9L, 1L, 6L, 1L, 3L, 1L, 1L, 7L, 9L, 1L, 1L, 1L, 1L, 
    8L, 9L, 2L, 2L, 1L, 1L, 1L, 4L, 1L, 2L, 1L, 1L, 2L, 3L, 2L, 
    2L, 9L, 1L, 1L, 2L, 6L, 1L, 4L, 2L, 1L, 4L, 1L, 1L, 1L, 5L, 
    1L, 9L, 1L, 6L, 3L, 4L, 5L, 1L, 2L, 2L, 1L, 1L, 1L, 4L, 1L, 
    3L, 3L, 1L, 2L, 2L, 7L, 2L, 1L, 2L, 2L, 3L, 6L, 1L, 2L, 1L, 
    4L, 1L, 3L, 3L, 1L, 9L, 2L, 4L, 2L, 1L, 2L, 5L, 2L, 2L, 9L, 
    1L, 2L, 2L, 1L, 1L, 6L, 2L, 4L, 4L, 4L, 4L, 4L, 3L, 4L, 7L, 
    1L, 2L, 9L, 1L, 2L, 1L, 1L, 1L, 1L, 2L, 1L, 9L, 9L, 2L, 3L, 
    5L, 3L, 2L, 5L, 6L, 2L, 1L, 4L, 6L, 2L, 6L, 1L, 4L, 9L, 1L, 
    4L, 2L, 1L, 6L, 2L, 9L, 6L, 1L, 1L, 1L, 1L, 4L, 9L, 1L, 2L, 
    3L, 1L, 1L, 1L, 6L, 1L, 1L, 3L, 2L, 5L, 2L, 9L, 4L, 2L, 9L, 
    2L, 4L, 2L, 2L, 5L, 1L, 4L, 2L, 9L, 5L, 2L, 9L, 1L, 9L, 6L, 
    2L, 1L, 4L, 4L, 1L, 6L, 9L, 9L, 2L, 1L, 3L, 1L, 1L, 2L, 9L, 
    1L, 1L, 2L, 1L, 1L, 6L, 3L, 9L, 9L, 4L, 9L, 2L, 5L, 9L, 1L, 
    2L, 9L, 2L, 6L, 1L, 4L, 1L, 2L, 9L, 2L, 1L, 4L, 9L, 3L, 4L, 
    1L, 1L, 2L, 2L, 2L, 9L, 1L, 2L, 3L, 4L, 6L, 5L, 1L, 2L, 9L, 
    2L, 4L, 1L, 2L, 2L, 1L, 6L, 5L, 9L, 2L, 2L, 2L, 6L, 9L, 2L, 
    1L, 1L, 6L, 1L, 5L, 9L, 4L, 1L, 1L, 3L, 4L, 9L, 8L, 2L, 4L, 
    1L, 1L, 4L, 4L, 3L, 2L, 2L, 9L, 5L, 2L, 1L, 1L, 3L, 2L, 2L, 
    6L, 1L, 2L, 1L, 2L, 1L, 1L, 2L, 4L, 3L, 1L, 1L, 9L, 2L, 1L, 
    4L, 6L, 3L, 3L, 2L, 9L, 2L, 1L, 5L, 6L, 4L, 6L, 2L, 1L, 2L, 
    4L, 6L, 4L, 9L, 1L, 2L, 9L, 1L, 9L, 3L, 1L, 2L, 3L, 9L, 2L, 
    9L, 1L, 1L, 2L, 6L, 7L, 1L, 2L, 5L, 5L, 6L, 1L, 1L, 1L, 4L, 
    3L, 2L, 2L, 1L, 6L, 4L, 9L, 2L, 9L, 8L, 1L, 4L, 1L, 3L, 1L, 
    2L, 1L, 1L, 5L, 3L, 2L, 1L, 1L, 1L, 3L, 4L, 2L, 5L, 1L, 2L, 
    1L, 8L, 1L, 1L, 1L, 3L, 9L, 3L, 3L, 9L, 9L, 2L, 2L, 2L, 1L, 
    9L, 1L, 3L, 4L, 5L, 4L, 1L, 2L, 2L, 2L, 1L, 5L, 1L, 3L, 3L, 
    9L, 2L, 2L, 6L, 4L, 2L, 2L, 2L, 2L, 2L, 9L, 3L, 9L, 3L, 1L, 
    1L, 6L, 5L, 2L, 2L, 1L, 3L, 2L, 1L, 4L, 2L, 1L, 2L, 9L, 3L, 
    2L, 1L, 2L, 2L, 2L, 6L, 1L, 5L, 1L, 2L, 3L, 4L, 5L, 8L, 8L, 
    4L, 2L, 2L, 3L, 5L, 1L, 4L, 1L, 6L, 1L, 3L, 1L, 2L, 2L, 3L, 
    2L, 9L, 4L, 2L, 1L, 3L, 2L, 2L, 2L, 2L, 2L, 9L, 2L, 9L, 9L, 
    2L, 2L, 5L, 1L, 1L, 1L, 2L, 1L, 2L, 2L, 1L, 4L, 1L, 1L, 1L, 
    1L, 6L, 4L, 2L, 1L, 2L, 6L, 1L, 3L, 1L, 2L, 1L, 1L, 1L, 2L, 
    1L, 2L, 1L, 1L, 9L, 1L, 3L, 1L, 9L, 3L, 2L, 9L, 1L, 1L, 1L, 
    1L, 2L, 1L, 6L, 2L, 2L, 1L, 8L, 1L, 1L, 9L, 1L, 9L, 6L, 2L, 
    4L, 1L, 2L, 4L, 2L, 5L, 1L, 1L, 6L, 2L, 2L, 1L, 2L, 9L, 6L, 
    1L, 3L, 2L, 1L, 1L, 2L, 9L, 2L, 1L, 2L, 8L, 1L, 9L, 2L, 4L, 
    1L, 2L, 4L, 1L, 3L, 4L, 1L, 9L, 9L, 1L, 2L, 9L, 1L, 2L, 1L, 
    2L, 4L, 9L, 6L, 1L, 9L, 9L, 2L, 2L, 9L, 2L, 9L, 1L, 6L, 7L, 
    1L, 1L, 2L, 9L, 1L, 1L, 1L, 5L, 1L, 4L, 2L, 9L, 2L, 9L, 2L, 
    1L, 2L, 4L, 1L, 9L, 6L, 9L, 5L, 4L, 5L, 9L, 2L, 5L, 3L, 1L, 
    2L, 3L, 9L, 6L, 6L, 8L, 2L, 2L, 9L, 3L, 2L, 1L, 1L, 1L, 1L, 
    5L, 1L, 2L, 1L, 1L, 6L, 2L, 6L, 2L, 2L, 3L, 1L, 6L, 9L, 8L, 
    3L, 2L, 1L, 1L, 2L, 2L, 1L, 9L, 1L, 1L, 6L, 9L, 1L, 2L, 6L, 
    2L, 1L, 2L, 2L, 5L, 6L, 6L, 5L, 7L, 1L, 1L, 9L, 2L, 5L, 2L, 
    1L, 8L, 9L, 1L, 3L, 2L, 2L, 2L, 3L, 1L, 4L, 1L, 6L, 2L, 5L, 
    1L, 1L, 4L, 3L, 7L, 2L, 3L, 1L, 1L, 1L, 4L, 2L, 1L, 1L, 2L, 
    1L, 2L, 2L, 6L, 9L, 3L, 1L, 2L, 2L, 3L, 2L, 3L, 1L, 2L, 1L, 
    2L, 5L, 1L, 1L, 1L, 1L, 1L, 3L, 4L, 1L, 2L, 2L, 1L, 1L, 1L, 
    1L, 4L, 6L, 1L, 2L, 1L, 1L, 4L, 2L, 2L, 2L, 9L, 1L, 2L, 1L, 
    9L, 5L, 4L, 4L, 9L, 2L, 1L, 9L, 6L, 3L, 1L, 1L, 2L, 3L, 1L, 
    9L, 9L, 2L, 2L, 7L, 7L, 9L, 1L, 3L, 1L, 1L, 2L, 4L, 4L, 9L, 
    2L, 2L, 3L, 9L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 1L, 6L, 1L, 2L, 
    1L, 1L, 3L, 4L, 2L, 2L, 1L, 6L, 2L, 4L, 2L, 1L, 3L, 4L, 3L, 
    2L, 2L, 2L, 1L, 2L, 1L, 1L, 1L, 1L, 6L, 1L, 9L, 9L, 2L, 2L, 
    2L, 3L, 8L, 3L, 2L, 1L, 2L, 1L, 9L, 2L, 5L, 1L, 1L, 2L, 1L, 
    2L, 2L, 6L, 4L, 1L, 3L, 1L, 2L, 2L, 9L, 1L, 3L, 8L, 2L, 9L, 
    1L, 1L, 1L, 2L, 1L, 4L, 1L, 4L, 3L, 1L, 1L, 1L, 1L, 1L, 4L, 
    1L, 2L, 2L, 6L, 1L, 4L, 2L, 1L, 2L, 1L, 1L, 2L, 9L, 1L, 1L, 
    1L, 2L, 1L, 1L, 5L, 2L, 2L, 2L, 6L, 6L, 3L, 5L, 1L, 2L, 1L, 
    5L, 3L, 1L, 2L, 1L, 1L, 1L, 2L, 2L, 1L, 1L, 1L, 5L, 2L, 2L, 
    3L, 1L, 2L, 6L, 4L, 1L, 1L, 2L, 1L, 1L, 6L, 4L, 3L, 2L, 1L, 
    9L, 2L, 2L, 2L, 2L, 9L, 1L, 1L, 9L, 9L, 2L, 5L, 2L, 7L, 3L, 
    6L, 1L, 1L, 2L, 2L, 2L, 5L, 1L, 2L, 1L, 3L, 1L, 1L, 2L, 9L, 
    1L, 3L, 2L, 2L, 4L, 1L, 3L, 1L, 9L, 2L, 5L, 1L, 1L, 4L, 2L, 
    2L, 6L, 2L, 1L, 6L, 6L, 1L, 2L, 1L, 4L, 1L, 2L, 1L, 1L, 1L, 
    2L, 2L, 4L, 5L, 9L, 1L, 7L, 2L, 4L, 1L, 4L, 6L, 2L, 1L, 1L, 
    3L, 2L, 3L, 1L, 2L, 2L, 2L, 5L, 6L, 4L, 1L, 1L, 1L, 3L, 4L, 
    6L, 2L, 9L, 1L, 6L, 4L, 2L, 3L, 2L, 2L, 5L, 9L, 3L, 4L, 9L, 
    1L, 3L, 6L, 9L, 1L, 2L, 3L, 1L, 2L, 2L, 9L, 1L, 2L, 5L, 2L, 
    2L, 6L, 2L, 1L, 1L, 2L, 6L, 2L, 1L, 1L, 2L, 9L, 4L, 1L, 3L, 
    1L, 9L, 2L, 9L, 3L, 6L, 9L, 2L, 1L, 9L, 1L, 2L, 5L, 4L, 9L, 
    4L, 2L, 9L, 2L, 2L, 3L, 2L, 1L, 2L, 1L, 2L, 3L, 1L, 1L, 2L, 
    2L, 1L, 4L, 1L, 1L, 4L, 4L, 9L, 2L, 2L, 4L, 1L, 9L, 9L, 2L, 
    3L, 3L, 1L, 2L, 9L, 9L, 1L, 9L, 2L, 3L, 1L, 8L, 1L, 1L, 3L, 
    2L, 2L, 2L, 1L, 1L, 9L, 3L, 5L, 2L, 4L, 1L, 8L, 2L, 2L, 6L, 
    2L, 2L, 9L, 2L, 1L, 1L, 2L, 1L, 2L, 3L, 1L, 2L, 4L, 5L, 1L, 
    4L, 4L, 2L, 4L, 2L, 9L, 8L, 6L, 6L, 2L, 1L, 3L, 4L, 4L, 1L, 
    1L, 4L, 2L, 1L, 9L, 1L, 1L, 5L, 1L, 9L, 1L, 6L, 1L, 3L, 1L, 
    2L, 6L, 9L, 4L, 9L, 2L, 1L, 2L, 3L, 1L, 1L, 2L, 2L, 1L, 6L, 
    4L, 1L, 6L, 4L, 1L, 1L, 9L, 1L, 9L, 8L, 1L, 2L, 1L, 9L, 3L, 
    1L, 9L, 1L, 1L, 4L, 3L, 3L, 2L, 2L, 1L, 1L, 1L, 6L, 5L, 3L, 
    2L, 3L, 1L, 2L, 2L, 1L, 1L, 9L, 1L, 2L, 2L, 1L, 1L, 1L, 3L, 
    4L, 9L, 9L, 2L, 1L, 1L, 1L, 9L, 1L, 1L, 1L, 2L, 9L, 1L, 1L, 
    1L, 9L, 1L, 2L, 1L, 1L, 1L, 4L, 3L, 4L, 1L, 1L, 6L, 1L, 3L, 
    9L, 4L, 9L, 1L, 2L, 1L, 4L, 1L, 3L, 2L, 1L, 2L, 2L, 1L, 2L, 
    2L, 1L, 4L, 1L, 3L, 5L, 1L, 3L, 1L, 8L, 2L, 2L, 9L, 1L, 2L, 
    6L, 1L, 3L, 9L, 1L, 4L, 1L, 5L, 4L, 3L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 4L, 2L, 2L, 6L, 1L, 6L, 4L, 3L, 2L, 2L, 2L, 9L, 5L, 
    9L, 2L, 9L, 1L, 4L, 4L, 4L, 5L, 4L, 2L, 1L, 1L, 2L, 3L, 9L, 
    4L, 2L, 1L, 9L, 1L, 2L, 2L, 7L, 3L, 1L, 4L, 2L, 6L, 1L, 5L, 
    1L, 1L, 4L, 6L, 2L, 6L, 1L, 8L, 1L, 5L, 3L, 5L, 4L, 1L, 3L, 
    2L, 4L, 1L, 9L, 9L, 2L, 4L, 1L, 1L, 9L, 1L, 3L, 3L, 1L, 1L, 
    3L, 1L, 2L, 2L, 2L, 1L, 1L, 1L, 9L, 2L, 9L, 4L, 4L, 4L, 1L, 
    9L, 1L, 4L, 2L, 4L, 9L, 9L, 1L, 1L, 4L, 9L, 2L, 1L, 2L, 1L, 
    2L, 3L, 9L, 4L, 4L, 3L, 1L, 9L, 2L, 2L, 2L, 4L, 1L, 2L, 1L, 
    3L, 5L, 5L, 1L, 4L, 2L, 6L, 2L, 4L, 3L, 4L, 4L, 1L, 9L, 1L, 
    1L, 5L, 5L, 9L, 1L, 3L, 4L, 1L, 1L, 2L, 5L, 2L, 9L, 2L, 2L, 
    2L, 4L, 2L, 2L, 3L, 2L, 9L, 2L, 2L, 2L, 1L, 1L, 2L, 4L, 1L, 
    6L, 8L, 1L, 1L, 1L, 3L, 7L, 1L, 1L, 2L, 3L, 3L, 1L, 6L, 2L, 
    1L, 1L, 1L, 2L, 6L, 1L, 1L, 3L, 3L, 4L, 5L, 1L, 4L, 2L, 2L, 
    2L, 4L, 2L, 2L, 1L, 1L, 1L, 1L, 9L, 2L, 2L, 2L, 2L, 2L, 2L, 
    1L, 1L, 9L, 9L, 1L, 2L, 3L, 5L, 2L, 1L, 2L, 1L, 2L, 4L, 2L, 
    2L, 2L, 1L, 8L, 2L, 6L, 1L, 3L, 2L, 2L, 1L, 6L, 3L, 3L, 1L, 
    1L, 1L, 3L, 9L, 3L, 2L, 6L, 4L, 6L, 1L, 8L, 3L, 4L, 2L, 4L, 
    1L, 2L, 5L, 2L, 1L, 3L, 6L, 4L, 1L, 5L, 1L, 4L, 3L, 2L, 4L, 
    2L, 4L, 6L, 1L, 4L, 3L, 2L, 9L, 1L, 4L, 1L, 2L, 5L, 4L, 4L, 
    9L, 1L, 1L, 1L, 6L, 2L, 2L, 6L, 2L, 6L, 4L, 1L, 9L, 4L, 2L, 
    2L, 1L, 2L, 2L, 2L, 3L, 2L, 9L, 5L, 2L, 7L, 1L, 1L, 5L, 2L, 
    9L, 1L, 4L, 9L, 9L, 2L, 1L, 3L, 1L, 6L, 5L, 9L, 9L, 2L, 2L, 
    4L, 9L, 6L, 2L, 1L, 3L, 4L, 9L, 8L, 5L, 1L, 4L, 1L, 2L, 6L, 
    2L, 4L, 3L, 3L, 2L, 8L, 1L, 1L, 3L, 9L, 1L, 9L, 4L, 9L, 4L, 
    2L, 2L, 2L, 2L, 2L, 1L, 9L, 1L, 2L, 9L, 1L, 2L, 2L, 2L, 9L, 
    4L, 6L, 9L, 1L, 6L, 3L, 1L, 2L, 9L, 3L, 6L, 3L, 4L, 2L, 1L, 
    2L, 1L, 2L, 6L, 2L, 2L, 4L, 7L, 1L, 8L, 6L, 9L, 2L, 2L, 9L, 
    1L, 6L, 1L, 2L, 2L, 2L, 1L, 4L, 9L, 9L, 9L, 4L, 5L, 2L, 9L, 
    4L, 1L, 2L, 5L, 1L, 1L, 9L, 1L, 1L, 5L, 3L, 1L, 2L, 1L, 6L, 
    2L, 1L), .Label = c("Con", "Lab", "LibDem", "UKIP", "Green", 
    "SNP", "PC", "Other", "None or DK"), class = "factor"), Dsexage1 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0), Dsexage2 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 
    1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), Dsexage3 = c(0, 0, 
    0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0, 1, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 
    1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 
    1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 1, 0, 
    0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 1, 0, 0, 1, 0, 
    0, 0, 0, 1, 1, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1, 0, 1, 0, 1, 
    0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 1, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 1, 1, 1, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 
    0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 
    0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 
    1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 
    1, 0, 0, 1, 0, 1, 0, 1, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 1, 1, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 1, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 
    0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 
    1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 
    0, 0, 1, 0, 1, 0, 1, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 
    0, 0, 0), Dsexage4 = c(0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 
    1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 1, 1, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 1, 
    1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    1, 1, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 
    1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 1, 1, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 
    1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 
    0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 1, 1, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 
    1, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 1), Dsexage5 = c(0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0), Dsexage6 = c(1, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 
    0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 
    0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 
    0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 
    1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0), Dsexage7 = c(0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 1, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 1, 0, 1, 1, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 
    0, 1, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 1, 
    1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 
    1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 
    0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 1, 0, 1, 1, 0, 0, 
    1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 
    0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 
    0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    1, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 1, 0, 1, 0, 1, 0, 
    0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 1, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 
    1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 1, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 1, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 1, 
    1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 1, 0, 1, 0, 1, 1, 0, 1, 0, 0, 1, 1, 0, 0, 
    1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 
    0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 0, 1, 
    0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 
    1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0), Dsexage8 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 
    1, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 1, 0, 1, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 1, 0, 0, 1, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 
    0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 
    0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 
    0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 1, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 
    0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 
    0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    1, 1, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 
    1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 
    0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 
    0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 
    1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 1, 1, 0, 0, 1, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0), Dregion1 = c(0, 1, 1, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 
    0, 0, 1, 1, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 
    0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 
    0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 
    1, 1, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 1, 0, 0, 0, 
    1, 1, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 0, 1, 1, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 1, 0, 0, 0, 1, 
    0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    1, 0, 0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 1, 0, 1, 1, 1, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 
    1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 
    0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 1, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 1, 0, 
    0, 0, 1, 0, 1, 1, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 1, 0, 1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 1, 1, 1, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 1, 0, 1, 1, 
    0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 1, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 1, 0, 1, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 1, 0, 
    1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 
    0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 
    0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 1, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 
    0, 1, 0, 1, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0), 
    Dregion2 = c(0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 
    0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 
    0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 
    0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 
    1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 
    1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 
    1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1, 
    0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), Dregion3 = c(0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 
    1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 
    0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), Dregion4 = c(0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 1, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 0), Dregion5 = c(0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    1, 0, 1, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 
    1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 1, 1, 
    0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 
    1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 
    1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 
    1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 1, 
    0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 
    1, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 1, 0, 
    1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 
    0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 
    1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 1, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 1, 1, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 
    0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 
    0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 
    0, 1, 0, 1, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 
    1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 
    0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 
    0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 1, 0, 1, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 
    0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 1, 0, 0, 1, 0, 
    0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 
    1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1), Dregion6 = c(0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 1, 1, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 
    0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 
    0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0), Dregion7 = c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 
    1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 
    0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 
    0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 1, 1, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 
    1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0)), row.names = c(NA, -2000L
), .Names = c("ID", "SexbyAge", "Region", "PartyID", "LTV", "VoteIntention", 
"Dsexage1", "Dsexage2", "Dsexage3", "Dsexage4", "Dsexage5", "Dsexage6", 
"Dsexage7", "Dsexage8", "Dregion1", "Dregion2", "Dregion3", "Dregion4", 
"Dregion5", "Dregion6", "Dregion7"), class = "data.frame")
